/*
 * Copyright (C) 2013 Square, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.squareup.wire.schema.internal;

import com.google.common.collect.ImmutableList;
import com.squareup.wire.schema.internal.parser.OptionElement;
import java.util.List;

public final class Util {
  public static final int MIN_TAG_VALUE = 1;
  public static final int MAX_TAG_VALUE = (1 << 29) - 1; // 536,870,911
  private static final int RESERVED_TAG_VALUE_START = 19000;
  private static final int RESERVED_TAG_VALUE_END = 19999;

  public static void appendDocumentation(StringBuilder builder, String documentation) {
    if (documentation.isEmpty()) {
      return;
    }
    for (String line : documentation.split("\n")) {
      builder.append("// ").append(line).append('\n');
    }
  }

  public static void appendOptions(StringBuilder builder, List<OptionElement> options) {
    int count = options.size();
    if (count == 1) {
      builder.append("[").append(options.get(0).toSchema()).append(']');
      return;
    }

    builder.append("[\n");
    for (int i = 0; i < count; i++) {
      String endl = (i < count - 1) ? "," : "";
      appendIndented(builder, options.get(i).toSchema() + endl);
    }
    builder.append(']');
  }

  public static void appendIndented(StringBuilder builder, String value) {
    for (String line : value.split("\n")) {
      builder.append("  ").append(line).append('\n');
    }
  }

  /** True if the supplied value is in the valid tag range and not reserved. */
  public static boolean isValidTag(int value) {
    return (value >= MIN_TAG_VALUE && value < RESERVED_TAG_VALUE_START)
        || (value > RESERVED_TAG_VALUE_END && value <= MAX_TAG_VALUE);
  }

  public static <T> ImmutableList<T> concatenate(List<T> a, T b) {
    return ImmutableList.<T>builder()
        .addAll(a)
        .add(b)
        .build();
  }

  private Util() {
  }
}
