/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.net.monitoring;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.twitter.common.base.MorePreconditions;
import com.twitter.common.net.loadbalancing.RequestTracker;
import com.twitter.common.net.monitoring.ConnectionMonitor;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import com.twitter.common.quantity.Unit;
import com.twitter.common.util.Clock;
import com.twitter.common.util.concurrent.ExecutorServiceShutdown;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;

public class TrafficMonitor<K>
implements ConnectionMonitor<K>,
RequestTracker<K> {
    @VisibleForTesting
    static final Amount<Long, Time> DEFAULT_GC_INTERVAL = Amount.of((long)5L, (Unit)Time.MINUTES);
    @GuardedBy(value="this")
    private final LoadingCache<K, TrafficInfo> trafficInfos;
    private final String serviceName;
    private final Amount<Long, Time> gcInterval;
    private AtomicLong lifetimeRequests = new AtomicLong();
    private final Clock clock;
    private final ScheduledExecutorService gcExecutor;

    public TrafficMonitor(String serviceName) {
        this(serviceName, DEFAULT_GC_INTERVAL);
    }

    public TrafficMonitor(String serviceName, Amount<Long, Time> gcInterval) {
        this(serviceName, gcInterval, Clock.SYSTEM_CLOCK);
    }

    public static <T> TrafficMonitor<T> create(String serviceName) {
        return new TrafficMonitor(serviceName);
    }

    @VisibleForTesting
    TrafficMonitor(String serviceName, Clock clock) {
        this(serviceName, DEFAULT_GC_INTERVAL, clock);
    }

    private TrafficMonitor(String serviceName, Amount<Long, Time> gcInterval, Clock clock) {
        this.serviceName = MorePreconditions.checkNotBlank((String)serviceName);
        this.clock = (Clock)Preconditions.checkNotNull((Object)clock);
        Preconditions.checkNotNull(gcInterval);
        Preconditions.checkArgument(((Long)gcInterval.getValue() > 0L ? 1 : 0) != 0, (Object)"GC interval must be > zero.");
        this.gcInterval = gcInterval;
        this.trafficInfos = CacheBuilder.newBuilder().build(new CacheLoader<K, TrafficInfo>(){

            public TrafficInfo load(K key) {
                return new TrafficInfo(key);
            }
        });
        Runnable gc = new Runnable(){

            @Override
            public void run() {
                TrafficMonitor.this.gc();
            }
        };
        this.gcExecutor = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("TrafficMonitor-gc-%d").build());
        this.gcExecutor.scheduleAtFixedRate(gc, (Long)gcInterval.as((Unit)Time.SECONDS), (Long)gcInterval.as((Unit)Time.SECONDS), TimeUnit.SECONDS);
    }

    public String getServiceName() {
        return this.serviceName;
    }

    public long getLifetimeRequestCount() {
        return this.lifetimeRequests.get();
    }

    public synchronized Map<K, TrafficInfo> getTrafficInfo() {
        return ImmutableMap.copyOf((Map)this.trafficInfos.asMap());
    }

    @Override
    public synchronized void connected(K key) {
        Preconditions.checkNotNull(key);
        ((TrafficInfo)this.trafficInfos.getUnchecked(key)).incConnections();
    }

    @Override
    public synchronized void released(K key) {
        Preconditions.checkNotNull(key);
        TrafficInfo info = (TrafficInfo)this.trafficInfos.getUnchecked(key);
        Preconditions.checkState((info.getConnectionCount() > 0 ? 1 : 0) != 0, (Object)"Double release detected!");
        info.decConnections();
    }

    @Override
    public void requestResult(K key, RequestTracker.RequestResult result, long requestTimeNanos) {
        Preconditions.checkNotNull(key);
        this.lifetimeRequests.incrementAndGet();
        ((TrafficInfo)this.trafficInfos.getUnchecked(key)).addResult(result);
    }

    @VisibleForTesting
    synchronized void gc() {
        Iterables.removeIf(this.trafficInfos.asMap().entrySet(), (Predicate)new Predicate<Map.Entry<K, TrafficInfo>>(){

            public boolean apply(Map.Entry<K, TrafficInfo> clientInfo) {
                if (clientInfo.getValue().connections.get() > 0) {
                    return false;
                }
                long idlePeriod = TrafficMonitor.this.clock.nowNanos() - clientInfo.getValue().getLastActiveTimestamp();
                return idlePeriod > (Long)TrafficMonitor.this.gcInterval.as((Unit)Time.NANOSECONDS);
            }
        });
    }

    public void shutdown() {
        new ExecutorServiceShutdown((ExecutorService)this.gcExecutor, Amount.of((long)0L, (Unit)Time.SECONDS)).execute();
    }

    public class TrafficInfo {
        private final K key;
        private AtomicInteger requestSuccesses = new AtomicInteger();
        private AtomicInteger requestFailures = new AtomicInteger();
        private AtomicInteger connections = new AtomicInteger();
        private AtomicLong lastActive = new AtomicLong();

        TrafficInfo(K key) {
            this.key = key;
            this.pulse();
        }

        void pulse() {
            this.lastActive.set(TrafficMonitor.this.clock.nowNanos());
        }

        public K getKey() {
            return this.key;
        }

        void addResult(RequestTracker.RequestResult result) {
            this.pulse();
            switch (result) {
                case SUCCESS: {
                    this.requestSuccesses.incrementAndGet();
                    break;
                }
                case FAILED: 
                case TIMEOUT: {
                    this.requestFailures.incrementAndGet();
                }
            }
        }

        public int getRequestSuccessCount() {
            return this.requestSuccesses.get();
        }

        public int getRequestFailureCount() {
            return this.requestFailures.get();
        }

        int incConnections() {
            this.pulse();
            return this.connections.incrementAndGet();
        }

        int decConnections() {
            this.pulse();
            return this.connections.decrementAndGet();
        }

        public int getConnectionCount() {
            return this.connections.get();
        }

        public long getLastActiveTimestamp() {
            return this.lastActive.get();
        }
    }
}

