/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.net.pool;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.twitter.common.base.Supplier;
import com.twitter.common.net.pool.Connection;
import com.twitter.common.net.pool.ConnectionFactory;
import com.twitter.common.net.pool.ObjectPool;
import com.twitter.common.net.pool.ResourceExhaustedException;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import com.twitter.common.quantity.Unit;
import com.twitter.common.stats.Stats;
import com.twitter.common.stats.StatsProvider;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class ConnectionPool<S extends Connection<?, ?>>
implements ObjectPool<S> {
    private static final Logger LOG = Logger.getLogger(ConnectionPool.class.getName());
    private final Set<S> leasedConnections = Sets.newSetFromMap((Map)Maps.newIdentityHashMap());
    private final Set<S> availableConnections = Sets.newHashSet();
    private final Lock poolLock;
    private final Condition available;
    private final ConnectionFactory<S> connectionFactory;
    private final Executor executor;
    private volatile boolean closed;
    private final AtomicLong connectionsCreated;
    private final AtomicLong connectionsDestroyed;
    private final AtomicLong connectionsReturned;

    public ConnectionPool(ConnectionFactory<S> connectionFactory) {
        this(connectionFactory, Stats.STATS_PROVIDER);
    }

    public ConnectionPool(ConnectionFactory<S> connectionFactory, StatsProvider statsProvider) {
        this(Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("CP-" + connectionFactory + "[%d]").setDaemon(true).build()), new ReentrantLock(true), connectionFactory, statsProvider);
    }

    @VisibleForTesting
    ConnectionPool(Executor executor, Lock poolLock, ConnectionFactory<S> connectionFactory, StatsProvider statsProvider) {
        Preconditions.checkNotNull((Object)executor);
        Preconditions.checkNotNull((Object)poolLock);
        Preconditions.checkNotNull(connectionFactory);
        Preconditions.checkNotNull((Object)statsProvider);
        this.executor = executor;
        this.poolLock = poolLock;
        this.available = poolLock.newCondition();
        this.connectionFactory = connectionFactory;
        String cfName = Stats.normalizeName((String)connectionFactory.toString());
        statsProvider.makeGauge("cp_leased_connections_" + cfName, (com.google.common.base.Supplier)new Supplier<Integer>(){

            public Integer get() {
                return ConnectionPool.this.leasedConnections.size();
            }
        });
        statsProvider.makeGauge("cp_available_connections_" + cfName, (com.google.common.base.Supplier)new Supplier<Integer>(){

            public Integer get() {
                return ConnectionPool.this.availableConnections.size();
            }
        });
        this.connectionsCreated = statsProvider.makeCounter("cp_created_connections_" + cfName);
        this.connectionsDestroyed = statsProvider.makeCounter("cp_destroyed_connections_" + cfName);
        this.connectionsReturned = statsProvider.makeCounter("cp_returned_connections_" + cfName);
    }

    public String toString() {
        return "CP-" + this.connectionFactory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public S get() throws ResourceExhaustedException, TimeoutException {
        this.checkNotClosed();
        this.poolLock.lock();
        try {
            Amount amount = this.leaseConnection((S)NO_TIMEOUT);
            return (S)amount;
        }
        finally {
            this.poolLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public S get(Amount<Long, Time> timeout) throws ResourceExhaustedException, TimeoutException {
        this.checkNotClosed();
        Preconditions.checkNotNull(timeout);
        if ((Long)timeout.getValue() == 0L) {
            return (S)this.get();
        }
        try {
            long start = System.nanoTime();
            long timeBudgetNs = (Long)timeout.as((Unit)Time.NANOSECONDS);
            if (!this.poolLock.tryLock(timeBudgetNs, TimeUnit.NANOSECONDS)) throw new TimeoutException("Timed out waiting for pool lock");
            try {
                Amount amount = this.leaseConnection((S)Amount.of((long)(timeBudgetNs -= System.nanoTime() - start), (Unit)Time.NANOSECONDS));
                return (S)amount;
            }
            finally {
                this.poolLock.unlock();
            }
        }
        catch (InterruptedException e) {
            throw new TimeoutException("Interrupted waiting for pool lock");
        }
    }

    private S leaseConnection(Amount<Long, Time> timeout) throws ResourceExhaustedException, TimeoutException {
        S connection = this.getConnection(timeout);
        if (connection == null) {
            throw new ResourceExhaustedException("Connection pool resources exhausted");
        }
        return this.leaseConnection(connection);
    }

    @Override
    public void release(S connection) {
        this.release(connection, false);
    }

    @Override
    public void remove(S connection) {
        this.release(connection, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void release(S connection, boolean remove) {
        this.poolLock.lock();
        try {
            if (!this.leasedConnections.remove(connection)) {
                throw new IllegalArgumentException("Connection not controlled by this connection pool: " + connection);
            }
            if (!this.closed && !remove && connection.isValid()) {
                this.addConnection(connection);
                this.connectionsReturned.incrementAndGet();
            } else {
                this.connectionFactory.destroy(connection);
                this.connectionsDestroyed.incrementAndGet();
            }
        }
        finally {
            this.poolLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.poolLock.lock();
        try {
            for (Connection availableConnection : this.availableConnections) {
                this.connectionFactory.destroy(availableConnection);
            }
        }
        finally {
            this.closed = true;
            this.poolLock.unlock();
        }
    }

    private void checkNotClosed() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
    }

    private S leaseConnection(S connection) {
        this.leasedConnections.add(connection);
        return connection;
    }

    private S getConnection(final Amount<Long, Time> timeout) throws ResourceExhaustedException, TimeoutException {
        if (this.availableConnections.isEmpty()) {
            if (this.leasedConnections.isEmpty()) {
                try {
                    return this.createConnection(timeout);
                }
                catch (Exception e) {
                    throw new ResourceExhaustedException("failed to create a new connection", e);
                }
            }
            if (this.connectionFactory.mightCreate()) {
                this.executor.execute(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            Connection connection = ConnectionPool.this.createConnection((Amount<Long, Time>)timeout);
                            if (connection != null) {
                                ConnectionPool.this.addConnection(connection);
                            } else {
                                LOG.log(Level.WARNING, "Failed to create a new connection for a waiting client due to maximum pool size or timeout");
                            }
                        }
                        catch (Exception e) {
                            LOG.log(Level.WARNING, "Failed to create a new connection for a waiting client", e);
                        }
                    }
                });
            }
            try {
                if ((Long)timeout.getValue() == 0L) {
                    while (this.availableConnections.isEmpty()) {
                        this.available.await();
                    }
                } else {
                    long timeRemainingNs = (Long)timeout.as((Unit)Time.NANOSECONDS);
                    while (this.availableConnections.isEmpty()) {
                        long start = System.nanoTime();
                        if (!this.available.await(timeRemainingNs, TimeUnit.NANOSECONDS)) {
                            throw new TimeoutException("timeout waiting for a connection to be released to the pool");
                        }
                        timeRemainingNs -= System.nanoTime() - start;
                    }
                    if (this.availableConnections.isEmpty()) {
                        throw new TimeoutException("timeout waiting for a connection to be released to the pool");
                    }
                }
            }
            catch (InterruptedException e) {
                throw new TimeoutException("Interrupted while waiting for a connection.");
            }
        }
        return this.getAvailableConnection();
    }

    private S getAvailableConnection() {
        Connection connection;
        Connection connection2 = connection = this.availableConnections.size() == 1 ? (Connection)Iterables.getOnlyElement(this.availableConnections) : (Connection)this.availableConnections.iterator().next();
        if (!this.availableConnections.remove(connection)) {
            throw new IllegalArgumentException("Connection picked not in pool: " + connection);
        }
        return (S)connection;
    }

    private S createConnection(Amount<Long, Time> timeout) throws Exception {
        S connection = this.connectionFactory.create(timeout);
        if (connection != null) {
            this.connectionsCreated.incrementAndGet();
        }
        return connection;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addConnection(S connection) {
        this.poolLock.lock();
        try {
            this.availableConnections.add(connection);
            this.available.signal();
        }
        finally {
            this.poolLock.unlock();
        }
    }
}

