/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.net.pool;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.twitter.common.base.Closure;
import com.twitter.common.base.Command;
import com.twitter.common.net.loadbalancing.LoadBalancer;
import com.twitter.common.net.loadbalancing.LoadBalancingStrategy;
import com.twitter.common.net.pool.Connection;
import com.twitter.common.net.pool.ObjectPool;
import com.twitter.common.net.pool.ResourceExhaustedException;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public class MetaPool<T, E>
implements ObjectPool<Connection<T, E>> {
    private final Command stopBackendRestorer;
    private Map<E, ObjectPool<Connection<T, E>>> backends = null;
    private final Lock backendsReadLock;
    private final Lock backendsWriteLock;
    private final Closure<Collection<E>> onBackendsChosen;
    private final LoadBalancer<E> loadBalancer;
    private static final Logger LOG = Logger.getLogger(MetaPool.class.getName());

    public MetaPool(LoadBalancer<E> loadBalancer, Closure<Collection<E>> onBackendsChosen, Amount<Long, Time> restoreInterval) {
        this(ImmutableMap.of(), loadBalancer, onBackendsChosen, restoreInterval);
    }

    public MetaPool(ImmutableMap<E, ObjectPool<Connection<T, E>>> backends, LoadBalancer<E> loadBalancer, Closure<Collection<E>> onBackendsChosen, Amount<Long, Time> restoreInterval) {
        this.loadBalancer = (LoadBalancer)Preconditions.checkNotNull(loadBalancer);
        this.onBackendsChosen = (Closure)Preconditions.checkNotNull(onBackendsChosen);
        ReentrantReadWriteLock backendsLock = new ReentrantReadWriteLock(true);
        this.backendsReadLock = backendsLock.readLock();
        this.backendsWriteLock = backendsLock.writeLock();
        this.setBackends((Map<E, ObjectPool<Connection<T, E>>>)backends);
        Preconditions.checkNotNull(restoreInterval);
        Preconditions.checkArgument(((Long)restoreInterval.getValue() > 0L ? 1 : 0) != 0);
        this.stopBackendRestorer = this.startDeadBackendRestorer(restoreInterval);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setBackends(Map<E, ObjectPool<Connection<T, E>>> pools) {
        this.backendsWriteLock.lock();
        try {
            this.backends = (Map)Preconditions.checkNotNull(pools);
            this.loadBalancer.offerBackends(pools.keySet(), this.onBackendsChosen);
        }
        finally {
            this.backendsWriteLock.unlock();
        }
    }

    private Command startDeadBackendRestorer(final Amount<Long, Time> restoreInterval) {
        final AtomicBoolean shouldRestore = new AtomicBoolean(true);
        Runnable restoreDeadBackends = new Runnable(){

            @Override
            public void run() {
                if (shouldRestore.get()) {
                    MetaPool.this.restoreDeadBackends((Amount<Long, Time>)restoreInterval);
                }
            }
        };
        final ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MTCP-DeadBackendRestorer[%s]").build());
        long restoreDelay = (Long)restoreInterval.getValue();
        scheduledExecutorService.scheduleWithFixedDelay(restoreDeadBackends, restoreDelay, restoreDelay, ((Time)restoreInterval.getUnit()).getTimeUnit());
        return new Command(){

            public void execute() {
                shouldRestore.set(false);
                scheduledExecutorService.shutdownNow();
                LOG.info("Backend restorer shut down");
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void restoreDeadBackends(Amount<Long, Time> restoreInterval) {
        for (E backend : this.snapshotBackends()) {
            ObjectPool<Connection<T, E>> pool;
            this.backendsReadLock.lock();
            try {
                pool = this.backends.get(backend);
            }
            finally {
                this.backendsReadLock.unlock();
            }
            if (pool == null) continue;
            try {
                this.release(this.get(backend, pool, restoreInterval));
            }
            catch (TimeoutException e) {
                LOG.warning("Backend restorer failed to revive backend: " + backend + " -> " + e);
            }
            catch (ResourceExhaustedException e) {
                LOG.warning("Backend restorer failed to revive backend: " + backend + " -> " + e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Iterable<E> snapshotBackends() {
        this.backendsReadLock.lock();
        try {
            ImmutableList immutableList = ImmutableList.copyOf(this.backends.keySet());
            return immutableList;
        }
        finally {
            this.backendsReadLock.unlock();
        }
    }

    @Override
    public Connection<T, E> get() throws ResourceExhaustedException, TimeoutException {
        return this.get((Amount)ObjectPool.NO_TIMEOUT);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Connection<T, E> get(Amount<Long, Time> timeout) throws ResourceExhaustedException, TimeoutException {
        ObjectPool<Connection<T, E>> pool;
        E backend;
        this.backendsReadLock.lock();
        try {
            backend = this.loadBalancer.nextBackend();
            Preconditions.checkNotNull(backend, (Object)"Load balancer gave a null backend.");
            pool = this.backends.get(backend);
            Preconditions.checkNotNull(backend, (String)"Given backend %s not found in tracked backends: %s", (Object[])new Object[]{backend, this.backends});
        }
        finally {
            this.backendsReadLock.unlock();
        }
        return this.get(backend, pool, timeout);
    }

    private Connection<T, E> get(E backend, ObjectPool<Connection<T, E>> pool, Amount<Long, Time> timeout) throws ResourceExhaustedException, TimeoutException {
        long startNanos = System.nanoTime();
        try {
            Connection<T, E> connection = (Long)timeout.getValue() == 0L ? pool.get() : pool.get(timeout);
            try {
                this.loadBalancer.connected(backend, System.nanoTime() - startNanos);
            }
            catch (RuntimeException e) {
                LOG.log(Level.WARNING, "Encountered an exception updating load balancer stats after leasing a connection - continuing", e);
            }
            return new ManagedConnection(connection, pool);
        }
        catch (TimeoutException e) {
            this.loadBalancer.connectFailed(backend, LoadBalancingStrategy.ConnectionResult.TIMEOUT);
            throw e;
        }
        catch (ResourceExhaustedException e) {
            this.loadBalancer.connectFailed(backend, LoadBalancingStrategy.ConnectionResult.FAILED);
            throw e;
        }
    }

    @Override
    public void release(Connection<T, E> connection) {
        this.release(connection, false);
    }

    @Override
    public void remove(Connection<T, E> connection) {
        this.release(connection, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void release(Connection<T, E> connection, boolean remove) {
        this.backendsWriteLock.lock();
        try {
            if (!(connection instanceof ManagedConnection)) {
                throw new IllegalArgumentException("Connection not controlled by this connection pool: " + connection);
            }
            ((ManagedConnection)connection).release(remove);
            this.loadBalancer.released(connection.getEndpoint());
        }
        finally {
            this.backendsWriteLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.stopBackendRestorer.execute();
        this.backendsWriteLock.lock();
        try {
            for (ObjectPool<Connection<T, E>> backend : this.backends.values()) {
                backend.close();
            }
        }
        finally {
            this.backendsWriteLock.unlock();
        }
    }

    private static class ManagedConnection<T, E>
    implements Connection<T, E> {
        private final Connection<T, E> connection;
        private final ObjectPool<Connection<T, E>> pool;

        private ManagedConnection(Connection<T, E> connection, ObjectPool<Connection<T, E>> pool) {
            this.connection = connection;
            this.pool = pool;
        }

        @Override
        public void close() {
            this.connection.close();
        }

        @Override
        public T get() {
            return this.connection.get();
        }

        @Override
        public boolean isValid() {
            return this.connection.isValid();
        }

        @Override
        public E getEndpoint() {
            return this.connection.getEndpoint();
        }

        public String toString() {
            return "ManagedConnection[" + this.connection.toString() + "]";
        }

        void release(boolean remove) {
            if (remove) {
                this.pool.remove(this.connection);
            } else {
                this.pool.release(this.connection);
            }
        }
    }
}

