/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.zookeeper;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.twitter.common.base.Command;
import com.twitter.common.base.Commands;
import com.twitter.common.base.ExceptionalSupplier;
import com.twitter.common.base.MorePreconditions;
import com.twitter.common.util.BackoffHelper;
import com.twitter.common.zookeeper.ZooKeeperClient;
import com.twitter.common.zookeeper.ZooKeeperUtils;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.ACL;

public class Group {
    private static final Logger LOG = Logger.getLogger(Group.class.getName());
    private static final Supplier<byte[]> NO_MEMBER_DATA = Suppliers.ofInstance(null);
    private static final String DEFAULT_NODE_NAME_PREFIX = "member_";
    private final ZooKeeperClient zkClient;
    private final ImmutableList<ACL> acl;
    private final String path;
    private final NodeScheme nodeScheme;
    private final Predicate<String> nodeNameFilter;
    private final BackoffHelper backoffHelper;

    public Group(ZooKeeperClient zkClient, Iterable<ACL> acl, String path, NodeScheme nodeScheme) {
        this.zkClient = (ZooKeeperClient)Preconditions.checkNotNull((Object)zkClient);
        this.acl = ImmutableList.copyOf(acl);
        this.path = ZooKeeperUtils.normalizePath((String)((String)Preconditions.checkNotNull((Object)path)));
        this.nodeScheme = (NodeScheme)Preconditions.checkNotNull((Object)nodeScheme);
        this.nodeNameFilter = new Predicate<String>(){

            public boolean apply(String nodeName) {
                return Group.this.nodeScheme.isMember(nodeName);
            }
        };
        this.backoffHelper = new BackoffHelper();
    }

    public Group(ZooKeeperClient zkClient, Iterable<ACL> acl, String path) {
        this(zkClient, acl, path, DEFAULT_NODE_NAME_PREFIX);
    }

    public Group(ZooKeeperClient zkClient, Iterable<ACL> acl, String path, String namePrefix) {
        this(zkClient, acl, path, new DefaultScheme(namePrefix));
    }

    public String getMemberPath(String memberId) {
        return this.path + "/" + MorePreconditions.checkNotBlank((String)memberId);
    }

    public String getPath() {
        return this.path;
    }

    public String getMemberId(String nodePath) {
        MorePreconditions.checkNotBlank((String)nodePath);
        Preconditions.checkArgument((boolean)nodePath.startsWith(this.path + "/"), (String)"Not a member of this group[%s]: %s", (Object[])new Object[]{this.path, nodePath});
        String memberId = StringUtils.substringAfterLast((String)nodePath, (String)"/");
        Preconditions.checkArgument((boolean)this.nodeScheme.isMember(memberId), (String)"Not a group member: %s", (Object[])new Object[]{memberId});
        return memberId;
    }

    public Iterable<String> getMemberIds() throws ZooKeeperClient.ZooKeeperConnectionException, KeeperException, InterruptedException {
        return Iterables.filter((Iterable)this.zkClient.get().getChildren(this.path, false), this.nodeNameFilter);
    }

    public byte[] getMemberData(String memberId) throws ZooKeeperClient.ZooKeeperConnectionException, KeeperException, InterruptedException {
        return this.zkClient.get().getData(this.getMemberPath(memberId), false, null);
    }

    public final Membership join() throws JoinException, InterruptedException {
        return this.join(NO_MEMBER_DATA, null);
    }

    public final Membership join(Supplier<byte[]> memberData) throws JoinException, InterruptedException {
        return this.join(memberData, null);
    }

    public final Membership join(@Nullable Command onLoseMembership) throws JoinException, InterruptedException {
        return this.join(NO_MEMBER_DATA, onLoseMembership);
    }

    public final Membership join(Supplier<byte[]> memberData, @Nullable Command onLoseMembership) throws JoinException, InterruptedException {
        Preconditions.checkNotNull(memberData);
        this.ensurePersistentGroupPath();
        final ActiveMembership groupJoiner = new ActiveMembership(memberData, onLoseMembership);
        return (Membership)this.backoffHelper.doUntilResult((ExceptionalSupplier)new ExceptionalSupplier<Membership, JoinException>(){

            public Membership get() throws JoinException {
                try {
                    return groupJoiner.join();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new JoinException("Interrupted trying to join group at path: " + Group.this.path, e);
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    LOG.log(Level.WARNING, "Temporary error trying to join group at path: " + Group.this.path, e);
                    return null;
                }
                catch (KeeperException e) {
                    if (Group.this.zkClient.shouldRetry(e)) {
                        LOG.log(Level.WARNING, "Temporary error trying to join group at path: " + Group.this.path, e);
                        return null;
                    }
                    throw new JoinException("Problem joining partition group at path: " + Group.this.path, e);
                }
            }
        });
    }

    private void ensurePersistentGroupPath() throws JoinException, InterruptedException {
        this.backoffHelper.doUntilSuccess((ExceptionalSupplier)new ExceptionalSupplier<Boolean, JoinException>(){

            public Boolean get() throws JoinException {
                try {
                    ZooKeeperUtils.ensurePath((ZooKeeperClient)Group.this.zkClient, (List)Group.this.acl, (String)Group.this.path);
                    return true;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new JoinException("Interrupted trying to ensure group at path: " + Group.this.path, e);
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    LOG.log(Level.WARNING, "Problem connecting to ZooKeeper, retrying", e);
                    return false;
                }
                catch (KeeperException e) {
                    if (Group.this.zkClient.shouldRetry(e)) {
                        LOG.log(Level.WARNING, "Temporary error ensuring path: " + Group.this.path, e);
                        return false;
                    }
                    throw new JoinException("Problem ensuring group at path: " + Group.this.path, e);
                }
            }
        });
    }

    public final void watch(GroupChangeListener groupChangeListener) throws WatchException, InterruptedException {
        Preconditions.checkNotNull((Object)groupChangeListener);
        try {
            this.ensurePersistentGroupPath();
        }
        catch (JoinException e) {
            throw new WatchException("Failed to create group path: " + this.path, e);
        }
        final GroupMonitor groupMonitor = new GroupMonitor(groupChangeListener);
        this.backoffHelper.doUntilSuccess((ExceptionalSupplier)new ExceptionalSupplier<Boolean, WatchException>(){

            public Boolean get() throws WatchException {
                try {
                    groupMonitor.watchGroup();
                    return true;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new WatchException("Interrupted trying to watch group at path: " + Group.this.path, e);
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    LOG.log(Level.WARNING, "Temporary error trying to watch group at path: " + Group.this.path, e);
                    return null;
                }
                catch (KeeperException e) {
                    if (Group.this.zkClient.shouldRetry(e)) {
                        LOG.log(Level.WARNING, "Temporary error trying to watch group at path: " + Group.this.path, e);
                        return null;
                    }
                    throw new WatchException("Problem trying to watch group at path: " + Group.this.path, e);
                }
            }
        });
    }

    public String toString() {
        return "Group " + this.path;
    }

    public static class DefaultScheme
    implements NodeScheme {
        private final String namePrefix;
        private final Pattern namePattern;

        public DefaultScheme(String namePrefix) {
            this.namePrefix = MorePreconditions.checkNotBlank((String)namePrefix);
            this.namePattern = Pattern.compile("^" + Pattern.quote(namePrefix) + "-?[0-9]+$");
        }

        @Override
        public boolean isMember(String nodeName) {
            return this.namePattern.matcher(nodeName).matches();
        }

        @Override
        public String createName(byte[] membershipData) {
            return this.namePrefix;
        }

        @Override
        public boolean isSequential() {
            return true;
        }
    }

    private class GroupMonitor {
        private final GroupChangeListener groupChangeListener;
        private Set<String> members;
        private final Watcher groupWatcher = new Watcher(){

            public final void process(WatchedEvent event) {
                if (event.getType() == Watcher.Event.EventType.NodeChildrenChanged) {
                    GroupMonitor.this.tryWatchGroup();
                }
            }
        };
        private final ExceptionalSupplier<Boolean, InterruptedException> tryWatchGroup = new ExceptionalSupplier<Boolean, InterruptedException>(){

            public Boolean get() throws InterruptedException {
                try {
                    GroupMonitor.this.watchGroup();
                    return true;
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    LOG.log(Level.WARNING, "Problem connecting to ZooKeeper, retrying", e);
                    return false;
                }
                catch (KeeperException e) {
                    if (Group.this.zkClient.shouldRetry(e)) {
                        LOG.log(Level.WARNING, "Temporary error re-watching group: " + Group.this.path, e);
                        return false;
                    }
                    throw new IllegalStateException("Permanent problem re-watching group: " + Group.this.path, e);
                }
            }
        };

        GroupMonitor(GroupChangeListener groupChangeListener) {
            this.groupChangeListener = groupChangeListener;
        }

        private void tryWatchGroup() {
            try {
                Group.this.backoffHelper.doUntilSuccess(this.tryWatchGroup);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(String.format("Interrupted while trying to re-watch group: %s, giving up", Group.this.path), e);
            }
        }

        private void watchGroup() throws ZooKeeperClient.ZooKeeperConnectionException, InterruptedException, KeeperException {
            List children = Group.this.zkClient.get().getChildren(Group.this.path, this.groupWatcher);
            this.setMembers(Iterables.filter((Iterable)children, (Predicate)Group.this.nodeNameFilter));
        }

        synchronized void setMembers(Iterable<String> members) {
            ImmutableSet membership;
            if (this.members == null) {
                Group.this.zkClient.registerExpirationHandler(new Command(){

                    public void execute() {
                        GroupMonitor.this.tryWatchGroup();
                    }
                });
            }
            if (!(membership = ImmutableSet.copyOf(members)).equals(this.members)) {
                this.groupChangeListener.onGroupChange(members);
                this.members = membership;
            }
        }
    }

    public static class WatchException
    extends Exception {
        public WatchException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static interface NodeScheme {
        public boolean isMember(String var1);

        public String createName(byte[] var1);

        public boolean isSequential();
    }

    public static interface GroupChangeListener {
        public void onGroupChange(Iterable<String> var1);
    }

    private class ActiveMembership
    implements Membership {
        private final Supplier<byte[]> memberData;
        private final Command onLoseMembership;
        private String nodePath;
        private String memberId;
        private volatile boolean cancelled;
        private byte[] membershipData;
        private final ExceptionalSupplier<Boolean, InterruptedException> tryJoin = new ExceptionalSupplier<Boolean, InterruptedException>(){

            public Boolean get() throws InterruptedException {
                try {
                    ActiveMembership.this.join();
                    return true;
                }
                catch (CancelledException e) {
                    return true;
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    LOG.log(Level.WARNING, "Problem connecting to ZooKeeper, retrying", e);
                    return false;
                }
                catch (KeeperException e) {
                    if (Group.this.zkClient.shouldRetry(e)) {
                        LOG.log(Level.WARNING, "Temporary error re-joining group: " + Group.this.path, e);
                        return false;
                    }
                    throw new IllegalStateException("Permanent problem re-joining group: " + Group.this.path, e);
                }
            }
        };

        public ActiveMembership(@Nullable Supplier<byte[]> memberData, Command onLoseMembership) {
            this.memberData = memberData;
            this.onLoseMembership = onLoseMembership == null ? Commands.NOOP : onLoseMembership;
        }

        @Override
        public String getGroupPath() {
            return Group.this.path;
        }

        @Override
        public synchronized String getMemberId() {
            return this.memberId;
        }

        @Override
        public synchronized String getMemberPath() {
            return this.nodePath;
        }

        @Override
        public synchronized byte[] updateMemberData() throws UpdateException {
            byte[] membershipData = (byte[])this.memberData.get();
            if (!ArrayUtils.isEquals((Object)this.membershipData, (Object)membershipData)) {
                try {
                    Group.this.zkClient.get().setData(this.nodePath, membershipData, -1);
                    this.membershipData = membershipData;
                }
                catch (KeeperException e) {
                    throw new UpdateException("Problem updating membership data.", e);
                }
                catch (InterruptedException e) {
                    throw new UpdateException("Interrupted attempting to update membership data.", e);
                }
                catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                    throw new UpdateException("Could not connect to the ZooKeeper cluster to update membership data.", e);
                }
            }
            return membershipData;
        }

        @Override
        public synchronized void cancel() throws JoinException {
            if (!this.cancelled) {
                try {
                    Group.this.backoffHelper.doUntilSuccess((ExceptionalSupplier)new ExceptionalSupplier<Boolean, JoinException>(){

                        public Boolean get() throws JoinException {
                            try {
                                Group.this.zkClient.get().delete(ActiveMembership.this.nodePath, -1);
                                return true;
                            }
                            catch (InterruptedException e) {
                                Thread.currentThread().interrupt();
                                throw new JoinException("Interrupted trying to cancel membership: " + ActiveMembership.this.nodePath, e);
                            }
                            catch (ZooKeeperClient.ZooKeeperConnectionException e) {
                                LOG.log(Level.WARNING, "Problem connecting to ZooKeeper, retrying", e);
                                return false;
                            }
                            catch (KeeperException.NoNodeException e) {
                                LOG.info("Membership already cancelled, node at path: " + ActiveMembership.this.nodePath + " has been deleted");
                                return true;
                            }
                            catch (KeeperException e) {
                                if (Group.this.zkClient.shouldRetry(e)) {
                                    LOG.log(Level.WARNING, "Temporary error cancelling membership: " + ActiveMembership.this.nodePath, e);
                                    return false;
                                }
                                throw new JoinException("Problem cancelling membership: " + ActiveMembership.this.nodePath, e);
                            }
                        }
                    });
                    this.cancelled = true;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new JoinException("Problem cancelling membership: " + this.nodePath, e);
                }
            }
        }

        synchronized Membership join() throws ZooKeeperClient.ZooKeeperConnectionException, InterruptedException, KeeperException {
            if (this.cancelled) {
                throw new CancelledException();
            }
            if (this.nodePath == null) {
                Group.this.zkClient.registerExpirationHandler(new Command(){

                    public void execute() {
                        ActiveMembership.this.tryJoin();
                    }
                });
            }
            byte[] membershipData = (byte[])this.memberData.get();
            String nodeName = Group.this.nodeScheme.createName(membershipData);
            CreateMode createMode = Group.this.nodeScheme.isSequential() ? CreateMode.EPHEMERAL_SEQUENTIAL : CreateMode.EPHEMERAL;
            this.nodePath = Group.this.zkClient.get().create(Group.this.path + "/" + nodeName, membershipData, (List)Group.this.acl, createMode);
            this.memberId = Group.this.getMemberId(this.nodePath);
            LOG.info("Set group member ID to " + this.memberId);
            this.membershipData = membershipData;
            Group.this.zkClient.get().exists(this.nodePath, new Watcher(){

                public void process(WatchedEvent event) {
                    if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                        LOG.info("Member ID deleted. Rejoining. Event: " + event);
                        ActiveMembership.this.tryJoin();
                    }
                }
            });
            return this;
        }

        private synchronized void tryJoin() {
            this.onLoseMembership.execute();
            try {
                Group.this.backoffHelper.doUntilSuccess(this.tryJoin);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(String.format("Interrupted while trying to re-join group: %s, giving up", Group.this.path), e);
            }
        }

        private class CancelledException
        extends IllegalStateException {
            private CancelledException() {
            }
        }
    }

    public static class UpdateException
    extends Exception {
        public UpdateException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class JoinException
    extends Exception {
        public JoinException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static interface Membership {
        public String getGroupPath();

        public String getMemberId();

        public String getMemberPath();

        public byte[] updateMemberData() throws UpdateException;

        public void cancel() throws JoinException;
    }
}

