package com.vaadin.copilot.analytics;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.base.devserver.stats.ProjectHelpers;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.CopilotVersion;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;

/**
 * Client for tracking features of Copilot
 */
public class AnalyticsClient extends CopilotServerClient {

    private static AnalyticsClient instance;

    private final String userKey;

    private final String machineId;

    private final boolean vaadiner;

    private boolean enabled;

    private Map<String, String> userContext;

    /**
     * Constructor loading ProKey
     */
    private AnalyticsClient() {
        this.userKey = ProjectHelpers.getUserKey();
        this.machineId = MachineId.get();
        ProKey proKey = LocalProKey.get();
        if (proKey != null) {
            UserInfo userInfo = new UserInfoServerClient().getUserInfo(proKey.getProKey());
            this.vaadiner = userInfo != null && userInfo.vaadiner();
        } else {
            this.vaadiner = false;
        }
    }

    /**
     * Get the singleton instance.
     *
     * @return the singleton instance of the client, never null
     */
    public static AnalyticsClient getInstance() {
        if (instance == null) {
            instance = new AnalyticsClient();
        }
        return instance;
    }

    /**
     * Set whether tracking is enabled.
     * <p>
     * Should only ever called from
     * {@link com.vaadin.copilot.CopilotIndexHtmlLoader}.
     *
     * @param enabled
     *            true if tracking should be enabled, false otherwise
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    /**
     * Check if tracking is enabled.
     *
     * @return true if tracking is enabled, false otherwise
     */
    public boolean isEnabled() {
        return enabled;
    }

    public void setUserContext(Map<String, String> userContext) {
        this.userContext = Collections.unmodifiableMap(userContext);
    }

    /**
     * Track event to copilot-server
     *
     * @param event
     *            Event name
     * @param properties
     *            Map of event properties
     */
    public void track(String event, Map<String, String> properties) {
        if (!isEnabled()) {
            return;
        }
        properties = properties == null ? new HashMap<>() : new HashMap<>(properties);
        properties.put("Vaadiner", String.valueOf(vaadiner));
        properties.put("Version", CopilotVersion.getVersion());
        sendRequest(event, properties, userContext);
    }

    protected void sendRequest(String event, Map<String, String> properties, Map<String, String> context) {
        AnalyticsRequest trackingRequest = new AnalyticsRequest(userKey, machineId, event, properties, context);
        String json = writeAsJsonString(trackingRequest);
        sendReactive("analytics", json);
    }

}
