package com.vaadin.copilot.javarewriter;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.copilot.ProjectManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasText;
import com.vaadin.flow.dom.ElementStateProvider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles the copy &amp; paste functionality for flow components.
 */
public class JavaRewriterCopyPasteHandler {

    private static final String VALUE_PROPERTY_KEY = "value";
    private static final String INVALID_PROPERTY_KEY = "invalid";
    private static final String ERROR_MESSAGE_PROPERTY_KEY = "errorMessage";

    private final ProjectManager projectManager;

    public JavaRewriterCopyPasteHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    /**
     * Collects required data for copying and pasting a component
     *
     * @param componentTypeAndSourceLocation
     *            Component type and source location with children info included
     * @return JavaComponent that will be used for pasting.
     */
    public JavaComponent getCopiedJavaComponent(ComponentTypeAndSourceLocation componentTypeAndSourceLocation) {
        Component component = componentTypeAndSourceLocation.component();
        ElementStateProvider stateProvider = component.getElement().getStateProvider();
        Map<String, Object> properties = stateProvider.getPropertyNames(component.getElement().getNode())
                .collect(Collectors.toMap(k -> k, k -> stateProvider.getProperty(component.getElement().getNode(), k)));

        addExtraProperties(component, properties);
        filterOutProperties(component, componentTypeAndSourceLocation.inheritanceChain(), properties);

        JavaComponent.Metadata copyMetadata = new JavaComponent.Metadata();
        try {
            copyMetadata = getCopyMetadata(componentTypeAndSourceLocation);
        } catch (Exception ex) {
            getLogger().error(ex.getMessage(), ex);
        }

        return new JavaComponent(getTag(componentTypeAndSourceLocation.inheritanceChain()),
                getFlowComponentClassName(componentTypeAndSourceLocation.inheritanceChain()), properties,
                componentTypeAndSourceLocation.children().stream().map(this::getCopiedJavaComponent).toList(),
                copyMetadata);
    }

    private String getFlowComponentClassName(List<Class> inheritanceChain) {
        for (Class aClass : inheritanceChain) {
            if (aClass.getName().startsWith("com.vaadin.flow")) {
                return aClass.getName();
            }
        }
        return inheritanceChain.get(0).getName();
    }

    /**
     * When a view is copied from the project, it should return the extended class
     * instead of view itself. //TODO in future this might change
     *
     * @param inheritanceChain
     *            inheritance chain of class
     * @return Tag name of the component e.g. class simple name
     */
    private String getTag(List<Class> inheritanceChain) {
        for (Class aClass : inheritanceChain) {
            if (aClass.getName().startsWith("com.vaadin.flow")) {
                return getTemplateTag(aClass.getSimpleName());
            }
        }
        return getTemplateTag(inheritanceChain.get(0).getSimpleName());
    }

    private String getTemplateTag(String originalTag) {
        if ("RadioButtonGroup".equals(originalTag)) {
            return "RadioGroup";
        }
        if ("CheckBoxItem".equals(originalTag)) {
            return "Checkbox";
        }
        return originalTag;
    }

    private JavaComponent.Metadata getCopyMetadata(ComponentTypeAndSourceLocation componentTypeAndSourceLocation)
            throws IOException {
        JavaComponent.Metadata metadata = new JavaComponent.Metadata();
        if (componentTypeAndSourceLocation.createLocation().className().startsWith("com.vaadin.flow.data")) {
            return metadata;
        }
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(componentTypeAndSourceLocation.javaFile()));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);

        Optional.ofNullable(componentInfo.localVariableName()).ifPresent(metadata::setLocalVariableName);
        Optional.ofNullable(componentInfo.fieldName()).ifPresent(metadata::setFieldVariableName);
        metadata.setOriginalClassName(componentTypeAndSourceLocation.component().getClass().getName());
        return metadata;
    }

    private void addExtraProperties(Component component, Map<String, Object> properties) {

        if (component.getElement().hasAttribute("disabled")) {
            properties.put("enabled", false);
        }

        if (component instanceof HasText hasText && !"".equals(hasText.getText())) {
            properties.put("text", hasText.getText());
        }

        // adding src and alt attributes as properties.
        if ("com.vaadin.flow.component.html.Image".equals(component.getClass().getName())) {
            if (!properties.containsKey("src") && component.getElement().hasAttribute("src")) {
                properties.put("src", component.getElement().getAttribute("src"));
            }
            if (!properties.containsKey("alt") && component.getElement().hasAttribute("alt")) {
                properties.put("alt", component.getElement().getAttribute("alt"));
            }
        }
        if ("com.vaadin.flow.component.radiobutton.RadioButton".equals(component.getClass().getName())
                && !properties.containsKey("label")) {
            component.getChildren()
                    .filter(child -> child.getClass().getName().equals("com.vaadin.flow.component.html.Label"))
                    .findFirst().ifPresent(label -> properties.put("label", label.getElement().getTextRecursively()));
        }
        if ("com.vaadin.flow.component.richtexteditor.RichTextEditor".equals(component.getClass().getName())) {
            getFieldValue(component, "currentMode")
                    .ifPresent(currentModeValue -> properties.put("valueChangeMode", currentModeValue));
        }
    }

    private void filterOutProperties(Component component, List<Class> inheritanceChain,
            Map<String, Object> properties) {
        boolean textFieldBase = inheritanceChain.stream()
                .anyMatch(clazz -> clazz.getName().equals("com.vaadin.flow.component.textfield.TextFieldBase"));
        boolean select = "com.vaadin.flow.component.select.Select".equals(component.getClass().getName());
        boolean checkboxGroup = "com.vaadin.flow.component.checkbox.CheckboxGroup"
                .equals(component.getClass().getName());
        boolean radioButtonGroup = "com.vaadin.flow.component.radiobutton.RadioButtonGroup"
                .equals(component.getClass().getName());
        boolean richTextEditor = "com.vaadin.flow.component.richtexteditor.RichTextEditor"
                .equals(component.getClass().getName());

        if (textFieldBase || select || checkboxGroup || radioButtonGroup) {
            // filtering out invalid properties that are generated in runtime for some
            // components
            properties.remove(VALUE_PROPERTY_KEY);
            properties.remove(INVALID_PROPERTY_KEY);
            properties.remove(ERROR_MESSAGE_PROPERTY_KEY);
            if (select) {
                properties.remove("opened");
            }
        }
        if (richTextEditor) {
            properties.remove("htmlValue");
            properties.remove(VALUE_PROPERTY_KEY);
        }
        if ("com.vaadin.flow.component.checkbox.CheckboxGroup$CheckBoxItem".equals(component.getClass().getName())) {
            properties.remove("checked");
            properties.remove("disabled");
        }
    }

    private Optional<Object> getFieldValue(Object target, String fieldName) {
        try {
            Field field = target.getClass().getDeclaredField(fieldName);
            field.setAccessible(true);
            Object o = field.get(target);
            return Optional.ofNullable(o);
        } catch (NoSuchFieldException ex) {
            getLogger().debug("Could not find field {} in class {}", fieldName, target.getClass().getName());
        } catch (IllegalAccessException e) {
            getLogger().debug("Could not access the field {}", fieldName, e);
        }
        return Optional.empty();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
