/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.util.Optional;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * Fetches an offline key from either the home directory (~/.vaadin/offlineKey)
 * or from a system property ({@code vaadin.offlineKey}) if available. Also
 * capable of writing an offline key but this should not be used except for
 * testing. Offline keys are always downloaded and stored manually.
 */
public class LocalOfflineKey {

    private static OfflineKey read(File offlineKeyLocation) throws IOException {
        if (!offlineKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(offlineKeyLocation)) {
            OfflineKey key = new OfflineKey(Util.toString(is));
            return key;
        } catch (ParseException e) {
            getLogger().error("Unable to read offline license from file", e);
            return null;
        }
    }

    public static File[] getLocations() {
        return Stream
                .of("offlineKey", "offlineKey.txt", "serverKey",
                        "serverKey.txt")
                .map(name -> new File(LocalProKey.getVaadinFolder(), name))
                .toArray(File[]::new);
    }

    /**
     * Finds an offline key, either from a system property or from the home
     * directory.
     *
     * @return an offline key or {@code null} if no key was found
     */
    public static OfflineKey get() {
        OfflineKey offlineKey = getSystemProperty();
        if (offlineKey != null) {
            getLogger().debug("Using offlineKey from system property");
            return offlineKey;
        }
        offlineKey = getEnvironmentVariable();
        if (offlineKey != null) {
            getLogger().debug("Using offlineKey from environment variable");
            return offlineKey;

        }

        try {
            File keyLocation = null;
            for (File location : getLocations()) {
                offlineKey = read(location);
                if (offlineKey != null) {
                    keyLocation = location;
                    break;
                }
            }

            if (offlineKey != null) {
                getLogger().debug("Found offline key in " + keyLocation);
            }
            return offlineKey;
        } catch (IOException e) {
            getLogger().debug("Unable to read offline key", e);
            return null;
        }
    }

    private static OfflineKey getSystemProperty() {
        String jwtData = System.getProperty("vaadin.offlineKey");
        if (jwtData == null) {
            jwtData = System.getProperty("vaadin.key");
            if (jwtData == null) {
                return null;
            }
        }

        try {
            return new OfflineKey(jwtData);
        } catch (ParseException e) {
            getLogger().error(
                    "Unable to read offline license from system property", e);
            return null;
        }
    }

    private static OfflineKey getEnvironmentVariable() {
        String value = EnvironmentVariables.get("VAADIN_OFFLINE_KEY");
        if (value == null) {
            value = EnvironmentVariables.get("VAADIN_KEY");
            if (value == null) {
                return null;
            }
        }
        try {
            return new OfflineKey(value);
        } catch (ParseException e) {
            getLogger().error(
                    "Unable to read offline license from environment variable",
                    e);
            return null;
        }
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
