/**
 * Copyright (C) 2012 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseChecker {

    private static boolean strictOffline = false;

    public interface Callback {

        public void ok();

        public void failed(Exception e);
    }

    static Consumer<String> systemBrowserUrlHandler = url -> {
        try {
            getLogger().info("Opening system browser to validate license. If the browser is not opened, please open "
                    + url + " manually");
            SystemBrowser.open(url);
        } catch (java.awt.HeadlessException e) {
            getLogger().error("You are running in a headless environment. Please open " + url
                    + " in any browser to validate the license.");
        } catch (IOException | URISyntaxException | UnsupportedOperationException e) {
            getLogger().error(
                    "Error opening system browser to validate license. Please open " + url + " manually", e);
        }

    };

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType)}
     */
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion) {
        checkLicenseFromStaticBlock(productName, productVersion, BuildType.DEVELOPMENT);
    }

    /**
     * Checks the license for the given product version from a {@code static}
     * block.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: production or development
     * 
     * @throws ExceptionInInitializerError if the license check fails
     */
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion, BuildType buildType) {
        try {
            checkLicense(productName, productVersion, buildType);
        } catch (Exception e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType)}
     */
    @Deprecated
    public static void checkLicense(String productName, String productVersion) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT, systemBrowserUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: development or production
     * 
     * @throws RuntimeException if the license check fails
     */
    public static void checkLicense(String productName, String productVersion, BuildType buildType) {
        checkLicense(productName, productVersion, buildType, systemBrowserUrlHandler);
    }

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType, Consumer)}
     */
    public static void checkLicense(String productName, String productVersion,
            Consumer<String> noKeyUrlHandler) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT, noKeyUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     * 
     * @param productName     the name of the product to check
     * @param productVersion  the version of the product to check
     * @param buildType       the type of build: development or production
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     * 
     * @throws RuntimeException if the license check fails
     */
    public static void checkLicense(String productName, String productVersion, BuildType buildType,
            Consumer<String> noKeyUrlHandler) {
        checkLicense(new Product(productName, productVersion), buildType, noKeyUrlHandler);
    }

    /**
     * @deprecated use
     *             {@link #checkLicenseAsync(String, String, BuildType, Callback)}
     */
    @Deprecated
    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback) {
        checkLicenseAsync(productName, productVersion, BuildType.DEVELOPMENT, callback);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: development or production
     * @param callback       the callback to invoke with the result
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback) {
        checkLicenseAsync(productName, productVersion, buildType, callback, systemBrowserUrlHandler);
    }

    /**
     * @deprecated use
     *             {@link #checkLicenseAsync(String, String, BuildType, Callback, Consumer)}
     */
    @Deprecated
    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback, Consumer<String> noKeyUrlHandler) {
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     * 
     * @param productName     the name of the product to check
     * @param productVersion  the version of the product to check
     * @param buildType       the type of build: development or production
     * @param callback        the callback to invoke with the result
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback, Consumer<String> noKeyUrlHandler) {
        new Thread(() -> {
            try {
                checkLicense(new Product(productName, productVersion), buildType, noKeyUrlHandler);
                callback.ok();
            } catch (Exception e) {
                callback.failed(e);
            }
        }).start();
    }

    /**
     * The internal method all license check methods end up calling.
     * <p>
     * Checking works like:
     * <ol>
     * <li>If there is a local pro key, check with the license server that the
     * proKey
     * has access to the given product
     * <li>If there is no local pro key but there is an offline key, check that this
     * offline key is valid and allows access to the given product
     * <li>If there was neither a local proKey nor an offline key, open the
     * vaadin.com URL for fetching a pro key for the user. Wait for the user to log
     * in, store the pro key and then validate it for the select product like in
     * step 1.
     * </ol>
     * 
     * @param product         the name and version of the product to check
     * @param buildType       the type of build: development or production
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     */
    private static void checkLicense(Product product, BuildType buildType, Consumer<String> noKeyUrlHandler) {
        getLogger().debug("Checking license for " + product);

        String machineId = MachineId.get();
        ProKey proKey = LocalProKey.get();

        if (proKey != null && ProKeyValidator.validate(product, proKey, machineId)) {
            // Online validation OK
            return;
        }

        OfflineKey offlineKey = LocalOfflineKey.get();
        if (offlineKey != null && OfflineKeyValidator.validate(product, buildType, offlineKey, machineId)) {
            // Offline validation OK
            return;
        }

        if (proKey == null) {
            // No proKey found - probably first launch

            proKey = VaadinComIntegration.openBrowserAndWaitForKey(product, noKeyUrlHandler);
            if (proKey != null) {
                LocalProKey.write(proKey);
                if (ProKeyValidator.validate(product, proKey, machineId)) {
                    // Online validation OK
                    return;
                }
            }
        }

        if (!strictOffline) {
            // Allow usage offline
            getLogger().warn(
                    "Unable to validate the license, please check your internet connection");
            return;
        }

        throw new RuntimeException(
                "Unable to validate the license, please check your internet connection. If you need to work offline then "
                        + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId));

    }

    public static Logger getLogger() {
        return LoggerFactory.getLogger(LicenseChecker.class);
    }

    public static void setStrictOffline(boolean strictOffline) {
        LicenseChecker.strictOffline = strictOffline;
    }

}
