/*
 * Copyright 2000-2022 Vaadin Ltd.
 *
 * Licensed under the Commercial Vaadin Developer License version 4.0 (CVDLv4);
 * you may not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * https://vaadin.com/license/cvdl-4.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.vaadin.client.ui;

import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style.Overflow;
import com.google.gwt.dom.client.Style.WhiteSpace;
import com.google.gwt.dom.client.TextAreaElement;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.user.client.DOM;
import com.vaadin.client.BrowserInfo;
import com.vaadin.client.WidgetUtil;
import com.vaadin.client.ui.dd.DragImageModifier;

/**
 * This class represents a multiline textfield (textarea).
 *
 * @author Vaadin Ltd.
 *
 */
@SuppressWarnings("deprecation")
public class VTextArea extends VTextField implements DragImageModifier {

    /** Default classname for this widget. */
    public static final String CLASSNAME = "v-textarea";

    private EnterDownHandler enterDownHandler = new EnterDownHandler();
    private boolean wordWrap = true;

    private class EnterDownHandler implements KeyDownHandler {
        @Override
        public void onKeyDown(KeyDownEvent event) {
            // Fix for #12424/13811 - if the key being pressed is enter, we stop
            // propagation of the KeyDownEvents if there were no modifier keys
            // also pressed. This prevents shortcuts that are bound to only the
            // enter key from being processed but allows usage of e.g.
            // shift-enter or ctrl-enter.
            if (event.getNativeKeyCode() == KeyCodes.KEY_ENTER
                    && !event.isAnyModifierKeyDown()) {
                event.stopPropagation();
            }
        }
    }

    /**
     * Constructs a widget for a TextArea.
     */
    public VTextArea() {
        super(DOM.createTextArea());
        setStyleName(CLASSNAME);
        addKeyDownHandler(enterDownHandler);
        getElement().getStyle().setOverflowX(Overflow.HIDDEN);
    }

    /**
     * Gets the base TextAreaElement of this widget.
     *
     * @return the base element
     */
    public TextAreaElement getTextAreaElement() {
        return super.getElement().cast();
    }

    /**
     * Sets the number of text rows that should be displayed.
     *
     * @param rows
     *            the number of text rows
     */
    public void setRows(int rows) {
        getTextAreaElement().setRows(rows);
    }

    /**
     * Sets whether the words should wrap or not.
     *
     * @param wordWrap
     *            {@code true} if the words should wrap, {@code false} otherwise
     */
    public void setWordWrap(boolean wordWrap) {
        if (wordWrap == this.wordWrap) {
            return;
        }
        if (wordWrap) {
            getElement().removeAttribute("wrap");
            getElement().getStyle().clearOverflowY();
            getElement().getStyle().setOverflowX(Overflow.HIDDEN);
            getElement().getStyle().clearWhiteSpace();
        } else {
            getElement().setAttribute("wrap", "off");
            getElement().getStyle().setOverflow(Overflow.AUTO);
            getElement().getStyle().setWhiteSpace(WhiteSpace.PRE);
        }
        if (BrowserInfo.get().isOpera()
                || (BrowserInfo.get().isWebkit() && wordWrap)) {
            // Opera fails to dynamically update the wrap attribute so we detach
            // and reattach the whole TextArea.
            // Webkit fails to properly reflow the text when enabling wrapping,
            // same workaround
            WidgetUtil.detachAttach(getElement());
        }
        this.wordWrap = wordWrap;
    }

    @Override
    public void modifyDragImage(Element element) {
        // Fix for #13557 - drag image doesn't show original text area text.
        // It happens because "value" property is not copied into the cloned
        // element
        String value = getElement().getPropertyString("value");
        if (value != null) {
            element.setPropertyString("value", value);
        }
    }
}
