/*
 * Copyright 2000-2022 Vaadin Ltd.
 *
 * Licensed under the Commercial Vaadin Developer License version 4.0 (CVDLv4);
 * you may not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * https://vaadin.com/license/cvdl-4.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.client.ui.slider;

import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.vaadin.client.communication.RpcProxy;
import com.vaadin.client.communication.StateChangeEvent;
import com.vaadin.client.ui.AbstractFieldConnector;
import com.vaadin.client.ui.VSlider;
import com.vaadin.client.ui.layout.ElementResizeListener;
import com.vaadin.shared.ui.Connect;
import com.vaadin.shared.ui.slider.SliderServerRpc;
import com.vaadin.shared.ui.slider.SliderState;
import com.vaadin.ui.Slider;

/**
 * A connector class for the Slider component.
 *
 * @author Vaadin Ltd
 */
@Connect(Slider.class)
public class SliderConnector extends AbstractFieldConnector
        implements ValueChangeHandler<Double> {

    /**
     * RPC instance for Slider's client-to-server calls.
     */
    protected SliderServerRpc rpc = RpcProxy.create(SliderServerRpc.class,
            this);

    private final ElementResizeListener resizeListener = event -> getWidget()
            .iLayout();

    @SuppressWarnings("deprecation")
    @Override
    public void init() {
        super.init();
        // The widget no longer uses the connection, but the value is still set
        // to ensure backwards compatibility.
        getWidget().setConnection(getConnection());
        getWidget().addValueChangeHandler(this);

        getLayoutManager().addElementResizeListener(getWidget().getElement(),
                resizeListener);
    }

    @Override
    public void onUnregister() {
        super.onUnregister();
        getLayoutManager().removeElementResizeListener(getWidget().getElement(),
                resizeListener);
    }

    @Override
    public VSlider getWidget() {
        return (VSlider) super.getWidget();
    }

    @Override
    public SliderState getState() {
        return (SliderState) super.getState();
    }

    @Override
    public void onValueChange(ValueChangeEvent<Double> event) {
        getState().value = event.getValue();
        rpc.valueChanged(event.getValue());
    }

    @SuppressWarnings("deprecation")
    @Override
    public void onStateChanged(StateChangeEvent stateChangeEvent) {
        super.onStateChanged(stateChangeEvent);

        // The widget no longer uses the connector id, but the value is still
        // set to ensure backwards compatibility.
        getWidget().setId(getConnectorId());
        getWidget().setDisabled(!isEnabled());
        getWidget().setReadOnly(isReadOnly());
        getWidget().setOrientation(getState().orientation);
        getWidget().setMinValue(getState().minValue);
        getWidget().setMaxValue(getState().maxValue);
        getWidget().setResolution(getState().resolution);
        getWidget().setValue(getState().value, false);
        getWidget().setUpdateValueOnClick(getState().updateValueOnClick);

        getWidget().buildBase();
        getWidget().setTabIndex(getState().tabIndex);
    }

}
