/*
 * Decompiled with CFR 0.152.
 */
package digital.pragmatech.testing;

import digital.pragmatech.testing.BeanCreationProfiler;
import digital.pragmatech.testing.ContextProfileData;
import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.ApplicationContextEvent;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.annotation.Order;

@Order(value=-2147483648)
public class TimingTrackingApplicationContextInitializer
implements ApplicationContextInitializer<ConfigurableApplicationContext> {
    private static final Logger logger = LoggerFactory.getLogger(TimingTrackingApplicationContextInitializer.class);
    private static final Map<ConfigurableApplicationContext, Instant> contextStartTimes = new ConcurrentHashMap<ConfigurableApplicationContext, Instant>();
    private static final Map<String, Long> contextLoadTimes = new ConcurrentHashMap<String, Long>();
    private static final Map<String, ContextProfileData> contextProfileData = new ConcurrentHashMap<String, ContextProfileData>();
    private static final Map<ConfigurableApplicationContext, Long> contextStartMemory = new ConcurrentHashMap<ConfigurableApplicationContext, Long>();

    public void initialize(ConfigurableApplicationContext applicationContext) {
        Instant startTime = Instant.now();
        long startMemory = this.getUsedMemory();
        String contextId = this.generateContextId(applicationContext);
        contextStartTimes.put(applicationContext, startTime);
        contextStartMemory.put(applicationContext, startMemory);
        ContextProfileData profileData = new ContextProfileData(contextId, startTime, startMemory);
        contextProfileData.put(contextId, profileData);
        logger.debug("Context profiling started for {} (start memory: {}MB)", (Object)contextId, (Object)(startMemory / 1024L / 1024L));
        BeanCreationProfiler beanProfiler = new BeanCreationProfiler(contextId);
        applicationContext.getBeanFactory().addBeanPostProcessor((BeanPostProcessor)beanProfiler);
        applicationContext.addBeanFactoryPostProcessor(beanFactory -> {
            profileData.setBeanDefinitionCount(beanFactory.getBeanDefinitionCount());
            profileData.recordPhase("BeanDefinitionRegistration", Instant.now());
            logger.debug("Registered {} bean definitions for context {}", (Object)beanFactory.getBeanDefinitionCount(), (Object)contextId);
        });
        applicationContext.addApplicationListener(event -> this.handleContextEvent(event, applicationContext, contextId, profileData, beanProfiler));
    }

    private void handleContextEvent(Object event, ConfigurableApplicationContext applicationContext, String contextId, ContextProfileData profileData, BeanCreationProfiler beanProfiler) {
        ApplicationContextEvent contextEvent;
        if (event instanceof ApplicationContextEvent && (contextEvent = (ApplicationContextEvent)event).getSource() == applicationContext) {
            String eventType = event.getClass().getSimpleName();
            profileData.recordPhase(eventType, Instant.now());
            if (event instanceof ContextRefreshedEvent) {
                this.finalizeContextProfiling(applicationContext, contextId, profileData, beanProfiler);
            }
            logger.debug("Context event {} for {}", (Object)eventType, (Object)contextId);
        }
    }

    private void finalizeContextProfiling(ConfigurableApplicationContext applicationContext, String contextId, ContextProfileData profileData, BeanCreationProfiler beanProfiler) {
        Instant endTime = Instant.now();
        long endMemory = this.getUsedMemory();
        Instant startTime = contextStartTimes.remove(applicationContext);
        Long startMemoryValue = contextStartMemory.remove(applicationContext);
        if (startTime != null) {
            long loadTimeMs = Duration.between(startTime, endTime).toMillis();
            long memoryUsed = endMemory - (startMemoryValue != null ? startMemoryValue : 0L);
            profileData.setEndTime(endTime);
            profileData.setEndMemory(endMemory);
            profileData.setTotalLoadTimeMs(loadTimeMs);
            profileData.setMemoryUsedMB(memoryUsed / 1024L / 1024L);
            profileData.setBeanCreationMetrics(beanProfiler.getMetrics());
            contextLoadTimes.put(contextId, loadTimeMs);
            logger.info("Context {} loaded in {}ms (memory: +{}MB, beans: {})", new Object[]{contextId, loadTimeMs, memoryUsed / 1024L / 1024L, beanProfiler.getMetrics().getTotalBeansCreated()});
        }
    }

    private String generateContextId(ConfigurableApplicationContext context) {
        return context.getClass().getSimpleName() + "@" + System.identityHashCode(context);
    }

    private long getUsedMemory() {
        Runtime runtime = Runtime.getRuntime();
        return runtime.totalMemory() - runtime.freeMemory();
    }

    public static ContextProfileData getContextProfileData(ConfigurableApplicationContext context) {
        String contextId = context.getClass().getSimpleName() + "@" + System.identityHashCode(context);
        return contextProfileData.get(contextId);
    }
}

