/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.spi.x509.revocation;

import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.model.x509.revocation.Revocation;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.x509.revocation.RepositoryRevocationSource;
import eu.europa.esig.dss.spi.x509.revocation.RevocationSource;
import eu.europa.esig.dss.spi.x509.revocation.RevocationToken;
import eu.europa.esig.dss.utils.Utils;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FileRevocationSource<R extends Revocation>
extends RepositoryRevocationSource<R> {
    private static final long serialVersionUID = 988823509470487135L;
    private static final Logger LOG = LoggerFactory.getLogger(FileRevocationSource.class);
    private static final String DEFAULT_REVOCATION_CACHE_SUBDIRECTORY = "/dss-cache-revocation";
    private File fileCacheDirectory = new File(System.getProperty("java.io.tmpdir"), "/dss-cache-revocation");

    protected FileRevocationSource() {
    }

    protected FileRevocationSource(RevocationSource<R> proxiedSource) {
        this.proxiedSource = proxiedSource;
    }

    public void setFileCacheDirectory(File fileCacheDirectory) {
        Objects.requireNonNull(fileCacheDirectory, "File cache directory cannot be null!");
        this.fileCacheDirectory = this.initializeCacheDirectory(fileCacheDirectory);
    }

    private File initializeCacheDirectory(File fileCacheDirectory) {
        if (!fileCacheDirectory.exists()) {
            if (fileCacheDirectory.mkdirs()) {
                LOG.info("Cache directory '{}' created successfully", (Object)fileCacheDirectory.getAbsolutePath());
            } else {
                throw new IllegalStateException(String.format("Unable to create cache directory '%s'", fileCacheDirectory.getAbsolutePath()));
            }
        }
        if (!fileCacheDirectory.isDirectory()) {
            throw new IllegalArgumentException(String.format("Cache path '%s' is not a directory", fileCacheDirectory.getAbsolutePath()));
        }
        return fileCacheDirectory;
    }

    public File getFileCacheDirectory() {
        return this.fileCacheDirectory;
    }

    @Override
    protected List<RevocationToken<R>> findRevocations(String key, CertificateToken certificateToken, CertificateToken issuerCertToken) {
        FileCacheEntry revocationCache = this.getRevocationCache(key);
        if (revocationCache.exists()) {
            try {
                RevocationToken<R> token = this.reconstructTokenFromEncodedData(revocationCache, certificateToken, issuerCertToken);
                if (token != null) {
                    return Collections.singletonList(token);
                }
                LOG.warn("Failed to reconstruct revocation token from cache for key: {}", (Object)key);
            }
            catch (Exception e) {
                LOG.warn("Failed to read revocation cache file for key '{}': {}", new Object[]{key, e.getMessage(), e});
            }
        }
        return Collections.emptyList();
    }

    private FileCacheEntry getRevocationCache(String key) {
        return new FileCacheEntry(key, this.getRevocationFileExtension());
    }

    @Override
    protected void insertRevocation(String revocationKey, RevocationToken<R> token) {
        FileCacheEntry revocationCache = this.getRevocationCache(revocationKey);
        this.saveRevocationToken(revocationCache, token);
        LOG.debug("Revocation token inserted into cache file for key: {}", (Object)revocationKey);
    }

    protected void saveRevocationToken(FileCacheEntry revocationCache, RevocationToken<R> token) {
        revocationCache.saveRevocationToken(token);
    }

    @Override
    protected void updateRevocation(String revocationKey, RevocationToken<R> token) {
        this.insertRevocation(revocationKey, token);
        LOG.debug("Revocation token updated in cache file for key: {}", (Object)revocationKey);
    }

    @Override
    protected void removeRevocation(String revocationKey) {
        FileCacheEntry revocationCache = this.getRevocationCache(revocationKey);
        if (revocationCache.exists()) {
            if (revocationCache.clean()) {
                LOG.debug("Revocation token and all associated data have been successfully removed from cache for key: {}", (Object)revocationKey);
            } else {
                LOG.debug("Some or none files associated with the revocation data have been removed for key: {}", (Object)revocationKey);
            }
        }
    }

    protected abstract RevocationToken<R> reconstructTokenFromEncodedData(FileCacheEntry var1, CertificateToken var2, CertificateToken var3);

    protected abstract String getRevocationFileExtension();

    protected void deleteCacheFile(File cacheFile) {
        try {
            Files.deleteIfExists(cacheFile.toPath());
        }
        catch (IOException e) {
            LOG.warn("Failed to delete cache file '{}': {}", (Object)cacheFile.getAbsolutePath(), (Object)e.getMessage());
        }
    }

    public void clearCache() {
        if (!this.fileCacheDirectory.exists()) {
            LOG.warn("Cache directory '{}' does not exist!", (Object)this.fileCacheDirectory.getAbsolutePath());
            return;
        }
        if (!this.fileCacheDirectory.isDirectory()) {
            LOG.warn("Cache directory '{}' is not a directory", (Object)this.fileCacheDirectory.getAbsolutePath());
            return;
        }
        try {
            Utils.cleanDirectory((File)this.fileCacheDirectory);
            LOG.info("Cache cleared for directory: {}", (Object)this.fileCacheDirectory.getAbsolutePath());
        }
        catch (IOException e) {
            LOG.warn("Failed to clean the directory '{}' : {}", new Object[]{this.fileCacheDirectory.getAbsolutePath(), e.getMessage(), e});
        }
    }

    protected class FileCacheEntry {
        private static final String CERT_FILE_EXTENSION = ".cer";
        private static final String URI_FILE_EXTENSION = ".uri";
        private final String key;
        private final String revocationExtension;

        public FileCacheEntry(String key, String revocationExtension) {
            this.key = key;
            this.revocationExtension = revocationExtension;
        }

        public byte[] getRevocationDataBinaries() {
            File cacheFile = this.getCacheRevocationFile();
            return this.getFileContent(cacheFile);
        }

        public String getRevocationDataSourceUrl() {
            File cacheUriFile = this.getCacheUriFile();
            byte[] fileContent = this.getFileContent(cacheUriFile);
            if (fileContent != null) {
                return new String(fileContent, StandardCharsets.UTF_8);
            }
            return null;
        }

        public CertificateToken getIssuerCertificateToken() {
            File issuerCertificateFile = this.getCacheRevocationIssuerCertificateFile();
            if (issuerCertificateFile.exists()) {
                try {
                    byte[] encodedCertificate = this.getFileContent(issuerCertificateFile);
                    return DSSUtils.loadCertificate(encodedCertificate);
                }
                catch (Exception e) {
                    LOG.warn("Unable to load revocation data issuer certificate from file with filename '{}' : {}", new Object[]{issuerCertificateFile.getName(), e.getMessage(), e});
                }
            }
            return null;
        }

        private byte[] getFileContent(File file) {
            try {
                if (file.exists()) {
                    return DSSUtils.toByteArray(file);
                }
                LOG.warn("The file '{}' does not exist or has been removed.", (Object)file.getName());
            }
            catch (Exception e) {
                LOG.warn("Failed to read revocation cache file for key '{}': {}", (Object)this.key, (Object)e.getMessage());
            }
            return null;
        }

        private File getCacheRevocationFile() {
            return this.getCacheFileFromKey(this.revocationExtension);
        }

        private File getCacheUriFile() {
            return this.getCacheFileFromKey(URI_FILE_EXTENSION);
        }

        private File getCacheRevocationIssuerCertificateFile() {
            return this.getCacheFileFromKey(CERT_FILE_EXTENSION);
        }

        private File getCacheFileFromKey(String fileExtension) {
            return new File(FileRevocationSource.this.fileCacheDirectory, this.key + fileExtension);
        }

        public void saveRevocationToken(RevocationToken<R> revocationToken) {
            Objects.requireNonNull(revocationToken, "RevocationToken cannot be null!");
            DSSUtils.saveToFile(revocationToken.getEncoded(), this.getCacheRevocationFile());
            if (revocationToken.getSourceURL() != null) {
                DSSUtils.saveToFile(revocationToken.getSourceURL().getBytes(StandardCharsets.UTF_8), this.getCacheUriFile());
            }
        }

        public void saveCertificateToken(CertificateToken certificateToken) {
            Objects.requireNonNull(certificateToken, "CertificateToken cannot be null!");
            DSSUtils.saveToFile(certificateToken.getEncoded(), this.getCacheRevocationIssuerCertificateFile());
        }

        public boolean clean() {
            File cacheCertificateFile;
            boolean cacheCleaned = this.removeFile(this.getCacheRevocationFile());
            File cacheUriFile = this.getCacheUriFile();
            if (cacheUriFile.exists()) {
                cacheCleaned ^= this.removeFile(cacheUriFile);
            }
            if ((cacheCertificateFile = this.getCacheRevocationIssuerCertificateFile()).exists()) {
                cacheCleaned ^= this.removeFile(cacheCertificateFile);
            }
            return cacheCleaned;
        }

        private boolean removeFile(File fileToRemove) {
            if (fileToRemove.exists()) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Deleting the file with filename '{}'...", (Object)fileToRemove.getName());
                }
                try {
                    Files.delete(fileToRemove.toPath());
                    return true;
                }
                catch (IOException e) {
                    String errorMessage = "Unable to remove the cached file with name '%s'. Reason : %s";
                    if (LOG.isDebugEnabled()) {
                        LOG.warn(String.format(errorMessage, fileToRemove.getName(), e.getMessage()), (Throwable)e);
                    } else {
                        LOG.warn(String.format(errorMessage, fileToRemove.getName(), e.getMessage()));
                    }
                    return false;
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Unable to remove the file with filename '{}'! The file does not exist.", (Object)fileToRemove.getName());
            }
            return false;
        }

        public boolean exists() {
            return this.getCacheRevocationFile().exists();
        }
    }
}

