/**
 * Copyright (C) 2011 Angelo Zerr <angelo.zerr@gmail.com> and Pascal Leclercq <pascal.leclercq@gmail.com>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free  of charge, to any person obtaining
 * a  copy  of this  software  and  associated  documentation files  (the
 * "Software"), to  deal in  the Software without  restriction, including
 * without limitation  the rights to  use, copy, modify,  merge, publish,
 * distribute,  sublicense, and/or sell  copies of  the Software,  and to
 * permit persons to whom the Software  is furnished to do so, subject to
 * the following conditions:
 *
 * The  above  copyright  notice  and  this permission  notice  shall  be
 * included in all copies or substantial portions of the Software.
 *
 * THE  SOFTWARE IS  PROVIDED  "AS  IS", WITHOUT  WARRANTY  OF ANY  KIND,
 * EXPRESS OR  IMPLIED, INCLUDING  BUT NOT LIMITED  TO THE  WARRANTIES OF
 * MERCHANTABILITY,    FITNESS    FOR    A   PARTICULAR    PURPOSE    AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE,  ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package fr.opensagres.xdocreport.template.textstyling.html;

import java.io.StringReader;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import fr.opensagres.xdocreport.core.logging.LogUtils;
import fr.opensagres.xdocreport.core.utils.StringUtils;
import fr.opensagres.xdocreport.template.textstyling.AbstractTextStylingTransformer;
import fr.opensagres.xdocreport.template.textstyling.IDocumentHandler;
import fr.opensagres.xdocreport.template.textstyling.ITextStylingTransformer;

/**
 * 
 * HTML text styling transformer to transform HTML to another document kind
 * (odt, docx, etc) syntax. The ODT, DOCX is represented with the given
 * {@link IDocumentHandler}.
 * 
 */
public class HTMLTextStylingTransformer extends AbstractTextStylingTransformer {

	public static final ITextStylingTransformer INSTANCE = new HTMLTextStylingTransformer();

	/**
	 * Logger for this class
	 */
	private static final Logger LOGGER = LogUtils
			.getLogger(HTMLTextStylingTransformer.class.getName());

	private static final String[] searchList = { "\r", "\n", "\t" };
	private static final String[] replacementList = { "", "", "" };

	@Override
	protected void doTransform(String content, IDocumentHandler documentHandler)
			throws Exception {
		// remove special characters \n, \r
		String xml = StringUtils.replaceEach(content, searchList,
				replacementList);
		// add root element if xml doesn't contaisn xml root element.
		xml = "<root>" + xml + "</root>";
		if (LOGGER.isLoggable(Level.FINE)) {
			LOGGER.fine(xml);
		}
		XMLReader xmlReader = XMLReaderFactory.createXMLReader();
		xmlReader.setContentHandler(new HTMLTextStylingContentHandler(
				documentHandler));
		xmlReader.parse(new InputSource(new StringReader(xml)));
	}

}
