/*
 * Decompiled with CFR 0.152.
 */
package io.activej.dns.protocol;

import io.activej.bytebuf.ByteBuf;
import io.activej.bytebuf.ByteBufPool;
import io.activej.common.exception.InvalidSizeException;
import io.activej.common.exception.MalformedDataException;
import io.activej.common.exception.UnknownFormatException;
import io.activej.dns.protocol.DnsQuery;
import io.activej.dns.protocol.DnsResourceRecord;
import io.activej.dns.protocol.DnsResponse;
import io.activej.dns.protocol.DnsTransaction;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import org.jetbrains.annotations.Nullable;

public final class DnsProtocol {
    private static final int MAX_SIZE = 512;
    private static final byte[] STANDARD_QUERY_HEADER = new byte[]{1, 0, 0, 1, 0, 0, 0, 0, 0, 0};
    private static final AtomicInteger xorshiftState = new AtomicInteger(1);

    public static short generateTransactionId() {
        return (short)xorshiftState.updateAndGet(old -> {
            short x = (short)old;
            x = (short)(x ^ (x & 0xFFFF) << 7);
            x = (short)(x ^ (x & 0xFFFF) >>> 9);
            x = (short)(x ^ (x & 0xFFFF) << 8);
            assert (x != 0) : "Xorshift LFSR can never produce zero";
            return x;
        });
    }

    public static ByteBuf createDnsQueryPayload(DnsTransaction transaction) {
        ByteBuf byteBuf = ByteBufPool.allocate((int)512);
        byteBuf.writeShort(transaction.getId());
        byteBuf.write(STANDARD_QUERY_HEADER);
        byte componentSize = 0;
        DnsQuery query = transaction.getQuery();
        byte[] domainBytes = query.getDomainName().getBytes(StandardCharsets.US_ASCII);
        int pos = -1;
        while (++pos < domainBytes.length) {
            if (domainBytes[pos] != 46) {
                componentSize = (byte)(componentSize + 1);
                continue;
            }
            byteBuf.writeByte(componentSize);
            byteBuf.write(domainBytes, pos - componentSize, (int)componentSize);
            componentSize = 0;
        }
        byteBuf.writeByte(componentSize);
        byteBuf.write(domainBytes, pos - componentSize, (int)componentSize);
        byteBuf.writeByte((byte)0);
        byteBuf.writeShort(query.getRecordType().getCode());
        byteBuf.writeShort(QueryClass.INTERNET.getCode());
        return byteBuf;
    }

    public static DnsResponse readDnsResponse(ByteBuf payload) throws MalformedDataException {
        try {
            short transactionId = payload.readShort();
            payload.moveHead(1);
            ResponseErrorCode errorCode = ResponseErrorCode.fromBits(payload.readByte() & 0xF);
            short questionCount = payload.readShort();
            int answerCount = payload.readShort();
            payload.moveHead(4);
            if (questionCount != 1) {
                throw new MalformedDataException("Received DNS response has question count not equal to one");
            }
            StringBuilder sb = new StringBuilder();
            byte componentSize = payload.readByte();
            while (componentSize != 0) {
                sb.append(new String(payload.array(), payload.head(), (int)componentSize, StandardCharsets.US_ASCII));
                payload.moveHead((int)componentSize);
                componentSize = payload.readByte();
                if (componentSize == 0) continue;
                sb.append('.');
            }
            String domainName = sb.toString();
            short recordTypeCode = payload.readShort();
            RecordType recordType = RecordType.fromCode(recordTypeCode);
            if (recordType == null) {
                throw new UnknownFormatException("Received DNS response with unknown query record type (" + Integer.toHexString(recordTypeCode & 0xFFFF) + ")");
            }
            short queryClassCode = payload.readShort();
            QueryClass queryClass = QueryClass.fromCode(queryClassCode);
            if (queryClass != QueryClass.INTERNET) {
                throw new UnknownFormatException("Received DNS response with unknown query class (" + Integer.toHexString(queryClassCode & 0xFFFF) + ")");
            }
            DnsQuery query = DnsQuery.of(domainName, recordType);
            DnsTransaction transaction = DnsTransaction.of(transactionId, query);
            if (errorCode != ResponseErrorCode.NO_ERROR) {
                return DnsResponse.ofFailure(transaction, errorCode);
            }
            ArrayList<InetAddress> ips = new ArrayList<InetAddress>();
            int minTtl = Integer.MAX_VALUE;
            for (int i = 0; i < answerCount; ++i) {
                byte b = payload.readByte();
                while (b != 0) {
                    int twoBits = (b & 0xFF) >> 6;
                    if (twoBits == 3) {
                        payload.moveHead(1);
                        break;
                    }
                    if (twoBits == 0) {
                        payload.moveHead((int)b);
                        b = payload.readByte();
                        continue;
                    }
                    throw new MalformedDataException("Unsupported compression method");
                }
                RecordType currentRecordType = RecordType.fromCode(payload.readShort());
                payload.moveHead(2);
                if (currentRecordType != recordType) {
                    payload.moveHead(4);
                    payload.moveHead((int)payload.readShort());
                    continue;
                }
                minTtl = Math.min(payload.readInt(), minTtl);
                short length = payload.readShort();
                if (length != recordType.dataLength) {
                    throw new InvalidSizeException("Bad record length received. " + (Object)((Object)recordType) + "-record length should be " + recordType.dataLength + " bytes, it was " + length);
                }
                byte[] bytes = new byte[length];
                payload.read(bytes);
                try {
                    ips.add(InetAddress.getByAddress(domainName, bytes));
                    continue;
                }
                catch (UnknownHostException unknownHostException) {
                    // empty catch block
                }
            }
            if (ips.isEmpty()) {
                return DnsResponse.ofFailure(transaction, ResponseErrorCode.NO_DATA);
            }
            return DnsResponse.of(transaction, DnsResourceRecord.of(ips.toArray(new InetAddress[0]), minTtl));
        }
        catch (IndexOutOfBoundsException e) {
            throw new MalformedDataException("Failed parsing DNS response", (Throwable)e);
        }
    }

    public static enum RecordType {
        A(1, 4),
        AAAA(28, 16);

        private final short code;
        private final short dataLength;

        private RecordType(int code, int dataLength) {
            this.code = (short)code;
            this.dataLength = (short)dataLength;
        }

        public short getCode() {
            return this.code;
        }

        public short getDataLength() {
            return this.dataLength;
        }

        @Nullable
        static RecordType fromCode(short code) {
            switch (code) {
                case 1: {
                    return A;
                }
                case 28: {
                    return AAAA;
                }
            }
            return null;
        }
    }

    public static enum QueryClass {
        INTERNET(1);

        private final short code;

        private QueryClass(int code) {
            this.code = (short)code;
        }

        public short getCode() {
            return this.code;
        }

        @Nullable
        public static QueryClass fromCode(short code) {
            if (code == 1) {
                return INTERNET;
            }
            return null;
        }
    }

    public static enum ResponseErrorCode {
        NO_ERROR,
        FORMAT_ERROR,
        SERVER_FAILURE,
        NAME_ERROR,
        NOT_IMPLEMENTED,
        REFUSED,
        NO_DATA,
        TIMED_OUT,
        UNKNOWN;


        static ResponseErrorCode fromBits(int rcodeBits) {
            if (rcodeBits < 0 || rcodeBits > 5) {
                return UNKNOWN;
            }
            return ResponseErrorCode.values()[rcodeBits];
        }
    }
}

