/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.archive;

import io.aeron.archive.Archive;
import io.aeron.archive.codecs.CatalogHeaderDecoder;
import io.aeron.archive.codecs.CatalogHeaderEncoder;
import io.aeron.archive.codecs.RecordingDescriptorDecoder;
import io.aeron.archive.codecs.RecordingDescriptorEncoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderDecoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderEncoder;
import io.aeron.protocol.DataHeaderFlyweight;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import org.agrona.BitUtil;
import org.agrona.BufferUtil;
import org.agrona.IoUtil;
import org.agrona.LangUtil;
import org.agrona.concurrent.EpochClock;
import org.agrona.concurrent.UnsafeBuffer;

class Catalog
implements AutoCloseable {
    static final long NULL_TIME = -1L;
    static final long NULL_POSITION = -1L;
    static final int PAGE_SIZE = 4096;
    static final int NULL_RECORD_ID = -1;
    static final int DEFAULT_RECORD_LENGTH = 1024;
    static final byte VALID = 1;
    static final byte INVALID = 0;
    static final int DESCRIPTOR_HEADER_LENGTH = 32;
    private static final int SCHEMA_VERSION = 0;
    private static final int DESCRIPTOR_BLOCK_LENGTH = 80;
    private final RecordingDescriptorHeaderDecoder descriptorHeaderDecoder = new RecordingDescriptorHeaderDecoder();
    private final RecordingDescriptorHeaderEncoder descriptorHeaderEncoder = new RecordingDescriptorHeaderEncoder();
    private final RecordingDescriptorEncoder descriptorEncoder = new RecordingDescriptorEncoder();
    private final RecordingDescriptorDecoder descriptorDecoder = new RecordingDescriptorDecoder();
    private final UnsafeBuffer indexUBuffer;
    private final MappedByteBuffer indexMappedBBuffer;
    private final int recordLength;
    private final int maxDescriptorStringsCombinedLength;
    private final int maxRecordingId;
    private final File archiveDir;
    private final int fileSyncLevel;
    private final EpochClock epochClock;
    private long nextRecordingId = 0L;

    Catalog(File archiveDir, FileChannel archiveDirChannel, int fileSyncLevel, EpochClock epochClock) {
        this(archiveDir, archiveDirChannel, fileSyncLevel, epochClock, true);
    }

    Catalog(File archiveDir, FileChannel archiveDirChannel, int fileSyncLevel, EpochClock epochClock, boolean fixOnRefresh) {
        this.archiveDir = archiveDir;
        this.fileSyncLevel = fileSyncLevel;
        this.epochClock = epochClock;
        File indexFile = new File(archiveDir, "archive.catalog");
        boolean indexPreExists = indexFile.exists();
        try (FileChannel channel = FileChannel.open(indexFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);){
            this.indexMappedBBuffer = channel.map(FileChannel.MapMode.READ_WRITE, 0L, Integer.MAX_VALUE);
            this.indexUBuffer = new UnsafeBuffer(this.indexMappedBBuffer);
            if (!indexPreExists && archiveDirChannel != null && fileSyncLevel > 0) {
                archiveDirChannel.force(fileSyncLevel > 1);
            }
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
        try {
            if (indexPreExists) {
                CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder().wrap(this.indexUBuffer, 0, 8, 0);
                if (catalogHeaderDecoder.version() != 0) {
                    throw new IllegalArgumentException("Catalog file version" + catalogHeaderDecoder.version() + " does not match software:" + 0);
                }
                this.recordLength = catalogHeaderDecoder.entryLength();
            } else {
                new CatalogHeaderEncoder().wrap(this.indexUBuffer, 0).version(0).entryLength(1024);
                this.recordLength = 1024;
            }
            this.maxDescriptorStringsCombinedLength = this.recordLength - 124;
            this.maxRecordingId = (Integer.MAX_VALUE - (2 * this.recordLength - 1)) / this.recordLength;
            this.refreshCatalog(fixOnRefresh);
        }
        catch (Throwable ex) {
            this.close();
            throw ex;
        }
    }

    @Override
    public void close() {
        IoUtil.unmap(this.indexMappedBBuffer);
    }

    long addNewRecording(long startPosition, long startTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        if (this.nextRecordingId > (long)this.maxRecordingId) {
            throw new IllegalStateException("Catalog is full, max recordings reached: " + this.maxRecordingId);
        }
        int combinedStringsLen = strippedChannel.length() + sourceIdentity.length() + originalChannel.length();
        if (combinedStringsLen > this.maxDescriptorStringsCombinedLength) {
            throw new IllegalArgumentException("Combined length of channel:'" + strippedChannel + "' and sourceIdentity:'" + sourceIdentity + "' and originalChannel:'" + originalChannel + "' exceeds max allowed:" + this.maxDescriptorStringsCombinedLength);
        }
        long newRecordingId = this.nextRecordingId++;
        this.indexUBuffer.wrap(this.indexMappedBBuffer, this.recordingDescriptorOffset(newRecordingId), this.recordLength);
        this.descriptorEncoder.wrap(this.indexUBuffer, 32);
        Catalog.initDescriptor(this.descriptorEncoder, newRecordingId, startTimestamp, startPosition, imageInitialTermId, segmentFileLength, termBufferLength, mtuLength, sessionId, streamId, strippedChannel, originalChannel, sourceIdentity);
        this.descriptorHeaderEncoder.wrap(this.indexUBuffer, 0).length(this.descriptorEncoder.encodedLength()).valid((byte)1);
        if (this.fileSyncLevel > 0) {
            this.indexMappedBBuffer.force();
        }
        return newRecordingId;
    }

    private int recordingDescriptorOffset(long newRecordingId) {
        return (int)(newRecordingId * (long)this.recordLength) + this.recordLength;
    }

    boolean wrapDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L || recordingId >= (long)this.maxRecordingId) {
            return false;
        }
        buffer.wrap(this.indexMappedBBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        this.descriptorHeaderDecoder.wrap(buffer, 0, 32, 0);
        return this.descriptorHeaderDecoder.length() != 0;
    }

    UnsafeBuffer wrapDescriptor(long recordingId) {
        UnsafeBuffer unsafeBuffer = new UnsafeBuffer();
        return this.wrapDescriptor(recordingId, unsafeBuffer) ? unsafeBuffer : null;
    }

    private void refreshCatalog(boolean fixOnRefresh) {
        if (fixOnRefresh) {
            this.forEach(this::refreshAndFixDescriptor);
        } else {
            this.forEach((headerEncoder, headerDecoder, descriptorEncoder, descriptorDecoder) -> ++this.nextRecordingId);
        }
    }

    void forEach(CatalogEntryProcessor consumer) {
        for (long recordingId = 0L; recordingId < (long)this.maxRecordingId && this.wrapDescriptor(recordingId, this.indexUBuffer); ++recordingId) {
            this.descriptorHeaderDecoder.wrap(this.indexUBuffer, 0, 32, 0);
            this.descriptorHeaderEncoder.wrap(this.indexUBuffer, 0);
            Catalog.wrapDescriptorDecoder(this.descriptorDecoder, this.indexUBuffer);
            this.descriptorEncoder.wrap(this.indexUBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
        }
    }

    boolean forEntry(CatalogEntryProcessor consumer, long recordingId) {
        if (this.wrapDescriptor(recordingId, this.indexUBuffer)) {
            this.descriptorHeaderDecoder.wrap(this.indexUBuffer, 0, 32, 0);
            this.descriptorHeaderEncoder.wrap(this.indexUBuffer, 0);
            Catalog.wrapDescriptorDecoder(this.descriptorDecoder, this.indexUBuffer);
            this.descriptorEncoder.wrap(this.indexUBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
            return true;
        }
        return false;
    }

    private void refreshAndFixDescriptor(RecordingDescriptorHeaderEncoder unused, RecordingDescriptorHeaderDecoder headerDecoder, RecordingDescriptorEncoder encoder, RecordingDescriptorDecoder decoder) {
        long recordingId = decoder.recordingId();
        if (headerDecoder.valid() == 1 && decoder.stopTimestamp() == -1L) {
            int segmentIndex = 0;
            File segmentFile = new File(this.archiveDir, Archive.segmentFileName(recordingId, segmentIndex));
            long startPosition = decoder.startPosition();
            if (!segmentFile.exists()) {
                encoder.stopPosition(startPosition);
            } else {
                File nextSegmentFile = new File(this.archiveDir, Archive.segmentFileName(recordingId, segmentIndex + 1));
                while (nextSegmentFile.exists()) {
                    segmentFile = nextSegmentFile;
                    nextSegmentFile = new File(this.archiveDir, Archive.segmentFileName(recordingId, ++segmentIndex + 1));
                }
                int segmentFileLength = decoder.segmentFileLength();
                long stopOffset = this.recoverStopOffset(segmentFile, segmentFileLength);
                int termBufferLength = decoder.termBufferLength();
                long recordingLength = startPosition % (long)termBufferLength + (long)(segmentIndex * segmentFileLength) + stopOffset;
                encoder.stopPosition(startPosition + recordingLength);
            }
            encoder.stopTimestamp(this.epochClock.time());
        }
        this.nextRecordingId = recordingId + 1L;
    }

    private long recoverStopOffset(File segmentFile, int segmentFileLength) {
        long lastFragmentSegmentOffset = 0L;
        try (FileChannel segment = FileChannel.open(segmentFile.toPath(), StandardOpenOption.READ);){
            ByteBuffer headerBB = BufferUtil.allocateDirectAligned(32, 32);
            DataHeaderFlyweight headerFlyweight = new DataHeaderFlyweight(headerBB);
            long nextFragmentSegmentOffset = 0L;
            do {
                headerBB.clear();
                if (32 != segment.read(headerBB, nextFragmentSegmentOffset)) {
                    throw new IllegalStateException("Unexpected read failure from file: " + segmentFile.getAbsolutePath() + " at position:" + nextFragmentSegmentOffset);
                }
                if (headerFlyweight.frameLength() == 0) break;
                lastFragmentSegmentOffset = nextFragmentSegmentOffset;
            } while ((nextFragmentSegmentOffset += (long)BitUtil.align(headerFlyweight.frameLength(), 32)) != (long)segmentFileLength);
            if (nextFragmentSegmentOffset / 4096L == lastFragmentSegmentOffset / 4096L) {
                lastFragmentSegmentOffset = nextFragmentSegmentOffset;
            }
        }
        catch (Exception ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        return lastFragmentSegmentOffset;
    }

    static void initDescriptor(RecordingDescriptorEncoder recordingDescriptorEncoder, long recordingId, long startTimestamp, long startPosition, int initialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        recordingDescriptorEncoder.recordingId(recordingId).startTimestamp(startTimestamp).stopTimestamp(-1L).startPosition(startPosition).stopPosition(-1L).initialTermId(initialTermId).segmentFileLength(segmentFileLength).termBufferLength(termBufferLength).mtuLength(mtuLength).sessionId(sessionId).streamId(streamId).strippedChannel(strippedChannel).originalChannel(originalChannel).sourceIdentity(sourceIdentity);
    }

    static void wrapDescriptorDecoder(RecordingDescriptorDecoder decoder, UnsafeBuffer descriptorBuffer) {
        decoder.wrap(descriptorBuffer, 32, 80, 0);
    }

    static void wrapDescriptorEncoder(RecordingDescriptorEncoder decoder, UnsafeBuffer descriptorBuffer) {
        decoder.wrap(descriptorBuffer, 32);
    }

    @FunctionalInterface
    static interface CatalogEntryProcessor {
        public void accept(RecordingDescriptorHeaderEncoder var1, RecordingDescriptorHeaderDecoder var2, RecordingDescriptorEncoder var3, RecordingDescriptorDecoder var4);
    }
}

