/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.hub.actions;

import io.confluent.connect.hub.actions.Installable;
import io.confluent.connect.hub.cli.ExitCode;
import io.confluent.connect.hub.exceptions.ConfluentHubClientException;
import io.confluent.connect.hub.io.Storage;
import io.confluent.connect.hub.platform.InstallationState;
import io.confluent.connect.hub.platform.InstallationTypeConsumer;
import io.confluent.connect.hub.rest.Repository;
import io.confluent.connect.hub.utils.IOUtils;
import io.confluent.connect.hub.utils.NamingUtils;
import io.confluent.pluginregistry.PluginId;
import io.confluent.pluginregistry.rest.entities.PluginLicense;
import io.confluent.pluginregistry.rest.entities.PluginManifest;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.function.Function;

public class ConfluentHubController {
    private static final String TMP_DIR_PREFIX = "confluent-hub-tmp";
    private static final String PLUGIN_PATH_CONFIG = "plugin.path";
    private static final String NOT_INSTALLED_COMPONENT_UNINSTALL_MSG = "Could not locate component, skipping uninstall request";
    private static final String UNABLE_TO_INSTALL_BUNDLED_COMPONENT_MSG = "Unable to install component because it's already bundled";
    private static final String UNINSTALL_CONFIRMATION_MSG_FORMAT = "Do you want to uninstall existing version %s?";
    private static final String ALREADY_INSTALLED_MSG = "Component is already installed. If you want to uninstall existing version, confirm or run the command with \"--no-prompt\" option";
    private static final String UNKNOWN_INSTALLATION_STATE_MSG = "Unknown installation state";
    private static final String CONFLUENTINC_OWNER_NAME = "confluentinc";
    private static final String THIRD_PARTY_CONFIRMATION_MSG_FORMAT = "You are about to install '%s' from %s, as published on Confluent Hub.";
    private static final String THIRD_PARTY_NOT_CONFIRMED_MSG = "Please confirm if you want to install this component from a third-party publisher.";
    private static final String LICENSE_NOT_ACCEPTED_MSG = "You have to accept a license agreement in order to install this component";
    private static final String UPDATING_WORKER_CONFIG_MSG = "Adding installation directory to plugin path in the following files:";
    private static final String NULL_MANIFEST_MSG = "Manifest can not be null";
    private static final String IS_NOT_AVAILABLE_FOR_INSTALLATION_MSG = "Component is not available for installation yet.";
    private final Storage storage;
    private final Repository hubRepository;

    public ConfluentHubController(Storage storage, Repository repository) {
        this.storage = storage;
        this.hubRepository = repository;
    }

    public ExitCode handle(Installable installable) {
        ExitCode exitCode;
        String component = installable.getComponent();
        String componentDir = installable.getComponentDir();
        if (this.storage.exists(component)) {
            exitCode = this.doInstallFromLocal(component, componentDir, installable.getForce(), installable.getThirdPartyConfirmation(), installable.getLicenseAcceptedConfirmation());
        } else {
            try {
                PluginId pluginId = NamingUtils.parsePluginId(component);
                exitCode = this.doInstallFromHub(pluginId, componentDir, installable.getForce(), installable.getThirdPartyConfirmation(), installable.getLicenseAcceptedConfirmation());
            }
            catch (ConfluentHubClientException e) {
                if (e.getCause() != null) {
                    IOUtils.error(e.getCause().getMessage(), new Object[0]);
                }
                IOUtils.error(e.getMessage(), new Object[0]);
                exitCode = e.getExitCode();
            }
        }
        if (exitCode.equals((Object)ExitCode.SUCCESSFUL_COMPLETION)) {
            return this.doUpdateWorkerConfigs(installable.getWorkerConfigs(), componentDir);
        }
        return exitCode;
    }

    protected ExitCode doUpdateWorkerConfigs(List<String> workerConfigs, String componentDir) {
        IOUtils.trace("Updating configuration files: {}", Arrays.toString(workerConfigs.toArray()));
        IOUtils.info(UPDATING_WORKER_CONFIG_MSG, new Object[0]);
        workerConfigs.forEach(configPath -> {
            IOUtils.indentedInfo(configPath, new Object[0]);
            List<String> value = this.storage.getPropertyValues((String)configPath, PLUGIN_PATH_CONFIG);
            if (!value.contains(componentDir)) {
                value.add(componentDir);
                this.storage.updateConfig((String)configPath, PLUGIN_PATH_CONFIG, value);
            }
        });
        return ExitCode.SUCCESSFUL_COMPLETION;
    }

    protected ExitCode doInstallFromLocal(String installFromFilePath, String installToPath, Function<String, Boolean> force, Function<String, Boolean> thirdPartyConfirmation, Function<PluginLicense, Boolean> licenseAcceptedConfirmation) {
        PluginManifest manifest = this.getLocalPluginManifest(installFromFilePath);
        this.ensureLicenseAccepted(manifest, licenseAcceptedConfirmation);
        this.ensureThirdPartyConfirmed(manifest, thirdPartyConfirmation);
        IOUtils.info("Installing a component {} {}, provided by {} from the local file: {} into directory: {}", new Object[]{manifest.getTitle(), manifest.getVersion(), manifest.getOwner().getName(), installFromFilePath, installToPath});
        InstallationState installationState = this.getInstallationType(manifest, installToPath);
        this.ensureInstallationType(manifest, installationState, installToPath, force);
        File componentFile = new File(installFromFilePath);
        String installToComponentFolder = NamingUtils.getInstallToComponentFolder(installToPath, manifest);
        File extractToDir = new File(installToComponentFolder);
        this.storage.unzip(componentFile, extractToDir);
        return ExitCode.SUCCESSFUL_COMPLETION;
    }

    private void ensureThirdPartyConfirmed(PluginManifest manifest, Function<String, Boolean> confirmation) {
        if (!CONFLUENTINC_OWNER_NAME.equals(manifest.getOwner().getUsername()) && !confirmation.apply(String.format(THIRD_PARTY_CONFIRMATION_MSG_FORMAT, manifest.getName(), manifest.getOwner().getName())).booleanValue()) {
            throw new ConfluentHubClientException(THIRD_PARTY_NOT_CONFIRMED_MSG, ExitCode.SECURITY_ISSUES);
        }
    }

    private void ensureInstallationType(PluginManifest manifest, InstallationState installationState, String installToPath, Function<String, Boolean> force) {
        if (InstallationState.BUNDLED == installationState) {
            throw new ConfluentHubClientException(UNABLE_TO_INSTALL_BUNDLED_COMPONENT_MSG, ExitCode.ALREADY_INSTALLED);
        }
        if (InstallationState.NOT_INSTALLED != installationState) {
            if (force.apply(String.format(UNINSTALL_CONFIRMATION_MSG_FORMAT, manifest.getVersion())).booleanValue()) {
                this.doUninstall(manifest, installationState, installToPath);
            } else {
                throw new ConfluentHubClientException(ALREADY_INSTALLED_MSG, ExitCode.ALREADY_INSTALLED);
            }
        }
    }

    private void ensureLicenseAccepted(PluginManifest manifest, Function<PluginLicense, Boolean> confirmation) {
        if (manifest.getLicenses() != null) {
            for (PluginLicense license : manifest.getLicenses()) {
                if (confirmation.apply(license).booleanValue()) continue;
                throw new ConfluentHubClientException(LICENSE_NOT_ACCEPTED_MSG, ExitCode.CANCELLED_BY_USER);
            }
        }
    }

    protected void doUninstall(PluginManifest manifest, InstallationState installationState, String path) {
        IOUtils.trace("Uninstalling {} from {}", manifest.getName(), path);
        switch (installationState) {
            case IMMEDIATE_DIRECTORY: {
                this.storage.delete(NamingUtils.getInstallToComponentFolder(path, manifest));
                break;
            }
            case UBER_JAR: {
                this.storage.delete(path);
                break;
            }
            case NOT_INSTALLED: {
                IOUtils.warn(NOT_INSTALLED_COMPONENT_UNINSTALL_MSG, new Object[0]);
                break;
            }
            default: {
                throw new IllegalArgumentException(UNKNOWN_INSTALLATION_STATE_MSG);
            }
        }
    }

    protected InstallationState getInstallationType(PluginManifest manifest, String installationPath) {
        if (!this.storage.exists(installationPath)) {
            return InstallationState.NOT_INSTALLED;
        }
        InstallationTypeConsumer countingConsumer = new InstallationTypeConsumer(manifest, installationPath);
        this.storage.list(installationPath, countingConsumer, false);
        IOUtils.trace("Installation state for the {} in the {} is {}", new Object[]{manifest.getName(), installationPath, countingConsumer.getInstallationState()});
        return countingConsumer.getInstallationState();
    }

    protected PluginManifest getLocalPluginManifest(String filePath) throws ConfluentHubClientException {
        IOUtils.trace("Getting manifest for a local component", new Object[0]);
        File extractToDir = this.storage.createTmpDir(TMP_DIR_PREFIX);
        File sourceFile = new File(filePath);
        this.storage.unzip(sourceFile, extractToDir);
        IOUtils.trace("Extracted local file from {} into {}", sourceFile.getPath(), extractToDir);
        PluginManifest manifest = this.storage.loadManifest(NamingUtils.getManifestPath(extractToDir.getPath()));
        this.storage.delete(extractToDir.getPath());
        return manifest;
    }

    protected ExitCode doInstallFromHub(PluginId pluginId, String installToPath, Function<String, Boolean> force, Function<String, Boolean> thirdPartyConfirmation, Function<PluginLicense, Boolean> licenseAcceptedConfirmation) {
        PluginManifest manifest = this.hubRepository.getManifest(NamingUtils.getPathForComponent(pluginId));
        this.ensureAvailableForInstallation(manifest);
        this.ensureLicenseAccepted(manifest, licenseAcceptedConfirmation);
        this.ensureThirdPartyConfirmed(manifest, thirdPartyConfirmation);
        IOUtils.info("Downloading component {} {}, provided by {} from Confluent Hub and installing into {}", new Object[]{manifest.getTitle(), manifest.getVersion(), manifest.getOwner().getName(), installToPath});
        InstallationState installationState = this.getInstallationType(manifest, installToPath);
        this.ensureInstallationType(manifest, installationState, installToPath, force);
        File extractToDir = this.storage.createTmpDir(TMP_DIR_PREFIX);
        String extractToFilePath = NamingUtils.getArchivePath(extractToDir.getPath(), manifest);
        URI archiveUri = NamingUtils.getArchiveUri(manifest.getArchive());
        try (InputStream is = this.hubRepository.getArchiveContent(archiveUri);){
            this.storage.verifiedCopy(is, extractToFilePath, manifest.getArchive());
        }
        catch (IOException e) {
            IOUtils.error(e.getMessage(), new Object[0]);
            return ExitCode.UNKNOWN_ERROR;
        }
        String installToComponentFolder = NamingUtils.getInstallToComponentFolder(installToPath, manifest);
        this.storage.createDirectories(installToPath);
        this.storage.unzip(new File(extractToFilePath), new File(installToComponentFolder));
        this.storage.delete(extractToDir.getPath());
        return ExitCode.SUCCESSFUL_COMPLETION;
    }

    private void ensureAvailableForInstallation(PluginManifest manifest) {
        if (manifest == null) {
            throw new ConfluentHubClientException(NULL_MANIFEST_MSG, ExitCode.IS_NOT_AVAILABLE_FOR_INSTALLATION);
        }
        if (manifest.getArchive() == null || manifest.getArchive().getName() == null) {
            throw new ConfluentHubClientException(IS_NOT_AVAILABLE_FOR_INSTALLATION_MSG, ExitCode.IS_NOT_AVAILABLE_FOR_INSTALLATION);
        }
    }
}

