/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.catalog.metrics;

import com.google.inject.Singleton;
import io.confluent.catalog.DataCatalogConfig;
import io.confluent.catalog.hook.SchemaAtlasTypes;
import io.confluent.catalog.model.ModelConstants;
import io.confluent.catalog.storage.EntitySearchService;
import io.confluent.catalog.storage.MetadataRegistry;
import io.confluent.catalog.util.CatalogTenantUtils;
import io.confluent.catalog.util.QualifiedNameGenerator;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistry;
import io.confluent.kafka.serializers.subject.TopicNameStrategy;
import io.confluent.rest.RestConfigException;
import java.io.Closeable;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.atlas.AtlasErrorCode;
import org.apache.atlas.annotation.GraphTransaction;
import org.apache.atlas.exception.AtlasBaseException;
import org.apache.atlas.model.TypeCategory;
import org.apache.atlas.model.instance.AtlasEntity;
import org.apache.atlas.model.instance.AtlasEntityHeader;
import org.apache.atlas.model.metrics.AtlasMetrics;
import org.apache.atlas.model.typedef.AtlasBusinessMetadataDef;
import org.apache.atlas.model.typedef.AtlasStructDef;
import org.apache.atlas.repository.Constants;
import org.apache.atlas.repository.graphdb.AtlasEdge;
import org.apache.atlas.repository.graphdb.AtlasEdgeDirection;
import org.apache.atlas.repository.graphdb.AtlasElement;
import org.apache.atlas.repository.graphdb.AtlasGraph;
import org.apache.atlas.repository.graphdb.AtlasIndexQuery;
import org.apache.atlas.repository.graphdb.AtlasVertex;
import org.apache.atlas.repository.store.graph.AtlasEntityStore;
import org.apache.atlas.repository.store.graph.v2.AtlasEntityStream;
import org.apache.atlas.repository.store.graph.v2.AtlasGraphUtilsV2;
import org.apache.atlas.repository.store.graph.v2.EntityGraphRetriever;
import org.apache.atlas.repository.store.graph.v2.EntityStream;
import org.apache.atlas.services.MetricsService;
import org.apache.atlas.type.AtlasEntityType;
import org.apache.atlas.type.AtlasStructType;
import org.apache.atlas.type.AtlasTypeRegistry;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class TenantMetricsService
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(MetricsService.class);
    private static final String TENANT_TAG_DEF_ENTITY_CACHE_FILENAME = "tenant_tag_def_entity.ser";
    public static final String TENANT_TYPE = "cf_tenant";
    public static final String TYPE = "type";
    public static final String TYPE_SUBTYPES = "typeAndSubTypes";
    public static final String ENTITY = "entity";
    public static final String TAG = "tag";
    public static final String BM = "businessMetadata";
    public static final String GENERAL = "general";
    protected static final String METRIC_COLLECTION_TIME = "collectionTime";
    protected static final String METRIC_TYPE_COUNT = "typeCount";
    protected static final String METRIC_TYPE_UNUSED_COUNT = "typeUnusedCount";
    protected static final String METRIC_ENTITY_COUNT = "entityCount";
    protected static final String METRIC_ENTITY_DELETED = "entityDeleted";
    protected static final String METRIC_ENTITY_ACTIVE = "entityActive";
    protected static final String METRIC_ENTITY_SHELL = "entityShell";
    protected static final String METRIC_TAG_COUNT = "tagCount";
    protected static final String METRIC_TAG_COUNT_PER_TYPE = "tagCountPerType";
    protected static final String METRIC_ENTITIES_PER_TAG = "tagEntities";
    protected static final String METRIC_BM_COUNT = "businessMetadataCount";
    protected static final String METRIC_BM_COUNT_PER_TYPE = "businessMetadataCountPerType";
    protected static final String METRIC_ENTITIES_PER_BM = "businessMetadataEntities";
    protected static final String METRIC_ATTRS_PER_BM = "businessMetadataAttrs";
    protected static final String METRIC_ENTITY_ACTIVE_INCL_SUBTYPES = "entityActive-typeAndSubTypes";
    protected static final String METRIC_ENTITY_DELETED_INCL_SUBTYPES = "entityDeleted-typeAndSubTypes";
    protected static final String METRIC_ENTITY_SHELL_INCL_SUBTYPES = "entityShell-typeAndSubTypes";
    private static final int ENCODE_BASE = 36;
    private static final String HASH_ALGORITHM = "MD5";
    private final Map<String, String> hashCache = new ConcurrentHashMap<String, String>();
    private final AtlasGraph atlasGraph;
    private final AtlasTypeRegistry typeRegistry;
    private final AtlasEntityStore entityStore;
    private final EntitySearchService searchService;
    private final EntityGraphRetriever entityGraphRetriever;
    private final String indexSearchPrefix = AtlasGraphUtilsV2.getIndexSearchPrefix();
    private final ConcurrentHashMap<String, Map<String, Map<String, Set<AtlasEntityHeader>>>> tenantTagDefEntitiesMap;
    private final boolean useAtlasSearchToCountTags;
    private final boolean countTagsTestingEnabled;
    private final Set<String> countTagsBySearchTenants;
    private final boolean countTagsPersistentCacheEnabled;
    private final String cacheDir;
    private final ReadWriteLock countTagsPersistentCacheLock;
    private final ExecutorService tagQueryThreadPool;
    private final ExecutorService tagSearchThreadPool;

    @Inject
    public TenantMetricsService(SchemaRegistry schemaRegistry, AtlasGraph graph, AtlasTypeRegistry typeRegistry, AtlasEntityStore entityStore, EntitySearchService searchService, EntityGraphRetriever entityGraphRetriever) {
        this.atlasGraph = graph;
        this.typeRegistry = typeRegistry;
        this.entityStore = entityStore;
        this.searchService = searchService;
        this.entityGraphRetriever = entityGraphRetriever;
        ConcurrentHashMap deserializedMap = null;
        try {
            DataCatalogConfig dataCatalogConfig = new DataCatalogConfig(schemaRegistry.config().originalProperties());
            this.useAtlasSearchToCountTags = dataCatalogConfig.countTagsUseAtlasSearch();
            this.countTagsTestingEnabled = dataCatalogConfig.countTagsTestingEnabled();
            if (this.countTagsTestingEnabled) {
                this.tagQueryThreadPool = Executors.newFixedThreadPool(10);
                this.tagSearchThreadPool = Executors.newFixedThreadPool(10);
            } else {
                this.tagQueryThreadPool = null;
                this.tagSearchThreadPool = null;
            }
            this.countTagsBySearchTenants = new HashSet<String>(dataCatalogConfig.getCountTagsTenantsUseAtlasSearch());
            this.cacheDir = dataCatalogConfig.getCatalogCacheDataDir();
            this.countTagsPersistentCacheEnabled = dataCatalogConfig.countTagsPersistentCacheEnabled();
        }
        catch (RestConfigException e) {
            throw new IllegalArgumentException("Could not instantiate TenantMetricsService", e);
        }
        if (this.countTagsPersistentCacheEnabled && this.cacheDir != null && !this.cacheDir.isEmpty()) {
            LOG.info("CountTagsPersistentCache enabled");
            try {
                Path dirPath = Paths.get(this.cacheDir, new String[0]);
                if (Files.notExists(dirPath, new LinkOption[0])) {
                    Files.createDirectories(dirPath, new FileAttribute[0]);
                }
                Path filePath = Paths.get(this.cacheDir, TENANT_TAG_DEF_ENTITY_CACHE_FILENAME);
                try (ObjectInputStream ois = new ObjectInputStream(Files.newInputStream(filePath, new OpenOption[0]));){
                    deserializedMap = (ConcurrentHashMap)ois.readObject();
                }
            }
            catch (Exception e) {
                LOG.error("Failed to deserialize tag entity cache", (Throwable)e);
            }
            Executors.newScheduledThreadPool(1).scheduleAtFixedRate(this::persistTenantTagDefEntitiesMap, 600L, 600L, TimeUnit.SECONDS);
        }
        this.tenantTagDefEntitiesMap = deserializedMap != null ? deserializedMap : new ConcurrentHashMap();
        this.countTagsPersistentCacheLock = new ReentrantReadWriteLock();
    }

    public List<String> getTenants() {
        return this.getTenants(AtlasEntity.Status.ACTIVE);
    }

    private List<String> getTenants(AtlasEntity.Status status) {
        Iterator result = null;
        String indexQuery = this.indexSearchPrefix + "\"" + Constants.ENTITY_TYPE_PROPERTY_KEY + "\" : (%s)" + " AND " + this.indexSearchPrefix + "\"" + Constants.STATE_PROPERTY_KEY + "\" : (%s)";
        indexQuery = String.format(indexQuery, TENANT_TYPE, status.name());
        AtlasEntityType cfEntity = this.typeRegistry.getEntityTypeByName(TENANT_TYPE);
        AtlasStructType.AtlasAttribute tenantAttr = cfEntity.getAttribute("tenant");
        try {
            result = this.atlasGraph.indexQuery("vertex_index", indexQuery).vertices();
        }
        catch (Exception e) {
            LOG.error("Failed fetching using indexQuery: " + e.getMessage());
        }
        ArrayList<AtlasVertex> entityVertices = new ArrayList<AtlasVertex>();
        this.getVerticesFromIndexQueryResult(result, entityVertices);
        return entityVertices.stream().map(v -> Objects.toString(EntityGraphRetriever.mapVertexToPrimitive((AtlasElement)v, (String)tenantAttr.getVertexPropertyName(), (AtlasStructDef.AtlasAttributeDef)tenantAttr.getAttributeDef()))).collect(Collectors.toList());
    }

    public void maybeCreateTenant(String tenant) throws AtlasBaseException {
        AtlasEntityType entityType = this.ensureEntityType(TENANT_TYPE);
        Map<String, String> attributes = Collections.singletonMap("qualifiedName", tenant);
        String guid = null;
        try {
            guid = this.entityStore.getGuidByUniqueAttributes(entityType, attributes);
        }
        catch (AtlasBaseException atlasBaseException) {
            // empty catch block
        }
        if (guid != null) {
            return;
        }
        AtlasEntity entity = new AtlasEntity(TENANT_TYPE);
        entity.setAttribute("qualifiedName", (Object)tenant);
        entity.setAttribute("tenant", (Object)tenant);
        entity.setAttribute("name", (Object)tenant);
        entity.setAttribute("nameLower", (Object)(tenant != null ? tenant.toLowerCase() : null));
        entity.setAttribute("createTime", (Object)new Date());
        this.entityStore.createOrUpdate((EntityStream)new AtlasEntityStream(entity), false);
    }

    private AtlasEntityType ensureEntityType(String typeName) throws AtlasBaseException {
        AtlasEntityType ret = this.typeRegistry.getEntityTypeByName(typeName);
        if (ret == null) {
            throw new AtlasBaseException(AtlasErrorCode.TYPE_NAME_INVALID, new String[]{TypeCategory.ENTITY.name(), typeName});
        }
        return ret;
    }

    public Optional<Double> getSchemaAttachRate(String tenant) {
        if (tenant == null || tenant.isEmpty()) {
            return Optional.empty();
        }
        AtomicLong topicTotal = new AtomicLong(0L);
        long schemaAttached = 0L;
        HashSet<String> attributes = new HashSet<String>();
        attributes.add("tenant");
        attributes.add("name");
        try {
            Map topicCount = this.searchService.searchAllNonDeprecatedEntities(ModelConstants.ENTITY_KAFKA_TOPIC, null, tenant, attributes, Collections.emptySet()).stream().filter(entity -> tenant.equals(entity.getAttribute("tenant"))).map(entity -> entity.getAttribute("name").toString()).peek(entity -> topicTotal.incrementAndGet()).collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
            if (topicTotal.get() == 0L) {
                return Optional.empty();
            }
            Set subjects = this.searchService.searchAllNonDeprecatedEntities(SchemaAtlasTypes.SR_SUBJECT_VERSION.getName(), null, tenant, attributes, Collections.emptySet()).stream().filter(entity -> tenant.equals(entity.getAttribute("tenant"))).map(entity -> entity.getAttribute("name").toString()).collect(Collectors.toSet());
            TopicNameStrategy topicNameStrategy = new TopicNameStrategy();
            for (Map.Entry entry : topicCount.entrySet()) {
                String topic = (String)entry.getKey();
                if (!subjects.contains(topicNameStrategy.subjectName(topic, false, null)) && !subjects.contains(topicNameStrategy.subjectName(topic, true, null))) continue;
                schemaAttached += entry.getValue().longValue();
            }
        }
        catch (AtlasBaseException e) {
            LOG.error("Failed to get schema attach rate for tenant {}", (Object)tenant, (Object)e);
            return Optional.empty();
        }
        return Optional.of((double)schemaAttached / (double)topicTotal.get());
    }

    @GraphTransaction
    public AtlasMetrics getMetrics(String tenant) {
        Collection entityDefNames = this.typeRegistry.getAllEntityDefNames();
        HashMap<Object, Long> activeEntityCount = new HashMap<Object, Long>();
        HashMap<Object, Long> deletedEntityCount = new HashMap<Object, Long>();
        HashMap<Object, Long> shellEntityCount = new HashMap<Object, Long>();
        Map<Object, Object> entityTypeBmCount = new HashMap();
        HashMap<String, Long> bmAttachedEntityCount = new HashMap<String, Long>();
        HashMap<String, Long> bmAttrCount = new HashMap<String, Long>();
        HashMap<String, Long> activeEntityCountTypeAndSubTypes = new HashMap<String, Long>();
        HashMap<String, Long> deletedEntityCountTypeAndSubTypes = new HashMap<String, Long>();
        HashMap<String, Long> shellEntityCountTypeAndSubTypes = new HashMap<String, Long>();
        AtlasEntityType cfEntity = this.typeRegistry.getEntityTypeByName(TENANT_TYPE);
        AtlasStructType.AtlasAttribute tenantAttr = cfEntity.getAttribute("tenant");
        long unusedTypeCount = 0L;
        long totalEntities = 0L;
        if (entityDefNames != null) {
            for (Object entityDefName : entityDefNames) {
                long shellCount;
                long deletedCount;
                long activeCount = this.getTypeCount((String)entityDefName, AtlasEntity.Status.ACTIVE, tenantAttr, tenant);
                if (activeCount > 0L) {
                    activeEntityCount.put(entityDefName, activeCount);
                    totalEntities += activeCount;
                }
                if ((deletedCount = this.getTypeCount((String)entityDefName, AtlasEntity.Status.DELETED, tenantAttr, tenant)) > 0L) {
                    deletedEntityCount.put(entityDefName, deletedCount);
                    totalEntities += deletedCount;
                }
                if (activeCount == 0L && deletedCount == 0L) {
                    ++unusedTypeCount;
                }
                if ((shellCount = this.getTypeShellCount((String)entityDefName, tenantAttr, tenant)) <= 0L) continue;
                shellEntityCount.put(entityDefName, shellCount);
            }
        }
        Collection allEntityTypes = this.typeRegistry.getAllEntityTypes();
        for (AtlasEntityType entityType : allEntityTypes) {
            long entityActiveCount = 0L;
            long entityDeletedCount = 0L;
            long entityShellCount = 0L;
            for (String type : entityType.getTypeAndAllSubTypes()) {
                entityActiveCount += activeEntityCount.get(type) == null ? 0L : (Long)activeEntityCount.get(type);
                entityDeletedCount += deletedEntityCount.get(type) == null ? 0L : (Long)deletedEntityCount.get(type);
                entityShellCount += shellEntityCount.get(type) == null ? 0L : (Long)shellEntityCount.get(type);
            }
            if (entityActiveCount > 0L) {
                activeEntityCountTypeAndSubTypes.put(entityType.getTypeName(), entityActiveCount);
            }
            if (entityDeletedCount > 0L) {
                deletedEntityCountTypeAndSubTypes.put(entityType.getTypeName(), entityDeletedCount);
            }
            if (entityShellCount <= 0L) continue;
            shellEntityCountTypeAndSubTypes.put(entityType.getTypeName(), entityShellCount);
        }
        Map<String, Map<String, Set<AtlasEntityHeader>>> tagDefEntityMap = this.countTags(tenant);
        Map<String, Set> countTagsByEntityType = tagDefEntityMap.values().stream().flatMap(innerMap -> innerMap.entrySet().stream()).collect(Collectors.toMap(Map.Entry::getKey, entry -> ((Set)entry.getValue()).stream().map(entity -> (String)entity.getAttribute("qualifiedName")).collect(Collectors.toSet()), (existingSet, newSet) -> {
            existingSet.addAll(newSet);
            return existingSet;
        }));
        Map<String, Long> entityTypeTagCount = countTagsByEntityType.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((Set)e.getValue()).size()));
        Map<String, Long> taggedEntityCount = tagDefEntityMap.entrySet().stream().collect(Collectors.toMap(outerEntry -> this.anonymize((String)outerEntry.getKey()), outerEntry -> ((Map)outerEntry.getValue()).values().stream().mapToLong(Set::size).sum()));
        Collection bmDefs = this.typeRegistry.getAllBusinessMetadataDefs();
        long bmDefCount = 0L;
        if (bmDefs != null) {
            String tenantPrefix = QualifiedNameGenerator.ensureTypeTenantPrefix(tenant, "");
            HashMap<String, Set<String>> countByEntityType = new HashMap<String, Set<String>>();
            for (AtlasBusinessMetadataDef bmDef : bmDefs) {
                if (!bmDef.getName().startsWith(tenantPrefix)) continue;
                ++bmDefCount;
                String typeTenantPrefix = CatalogTenantUtils.tenantToEscaped(tenant) + "__";
                String anonymizedBMDefName = this.anonymize(bmDef.getName().replace(typeTenantPrefix, ""));
                List attrDefs = bmDef.getAttributeDefs();
                bmAttrCount.put(anonymizedBMDefName, Long.valueOf(attrDefs == null ? 0 : attrDefs.size() - 1));
                this.countBM(bmDef.getName(), anonymizedBMDefName, tenantAttr, tenant, countByEntityType, bmAttachedEntityCount);
            }
            entityTypeBmCount = countByEntityType.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((Set)e.getValue()).size()));
        }
        AtlasMetrics metrics = new AtlasMetrics();
        metrics.addMetric(GENERAL, METRIC_COLLECTION_TIME, (Object)System.currentTimeMillis());
        metrics.addMetric(GENERAL, METRIC_TYPE_COUNT, (Object)this.getAllTypesCount());
        metrics.addMetric(GENERAL, METRIC_TAG_COUNT, (Object)tagDefEntityMap.keySet().size());
        metrics.addMetric(GENERAL, METRIC_BM_COUNT, (Object)bmDefCount);
        metrics.addMetric(GENERAL, METRIC_TYPE_UNUSED_COUNT, (Object)unusedTypeCount);
        metrics.addMetric(GENERAL, METRIC_ENTITY_COUNT, (Object)totalEntities);
        metrics.addMetric(ENTITY, METRIC_ENTITY_ACTIVE, activeEntityCount);
        metrics.addMetric(ENTITY, METRIC_ENTITY_DELETED, deletedEntityCount);
        metrics.addMetric(ENTITY, METRIC_ENTITY_SHELL, shellEntityCount);
        metrics.addMetric(ENTITY, METRIC_ENTITY_ACTIVE_INCL_SUBTYPES, activeEntityCountTypeAndSubTypes);
        metrics.addMetric(ENTITY, METRIC_ENTITY_DELETED_INCL_SUBTYPES, deletedEntityCountTypeAndSubTypes);
        metrics.addMetric(ENTITY, METRIC_ENTITY_SHELL_INCL_SUBTYPES, shellEntityCountTypeAndSubTypes);
        metrics.addMetric(TAG, METRIC_TAG_COUNT_PER_TYPE, entityTypeTagCount);
        metrics.addMetric(TAG, METRIC_ENTITIES_PER_TAG, taggedEntityCount);
        metrics.addMetric(BM, METRIC_BM_COUNT_PER_TYPE, entityTypeBmCount);
        metrics.addMetric(BM, METRIC_ENTITIES_PER_BM, bmAttachedEntityCount);
        metrics.addMetric(BM, METRIC_ATTRS_PER_BM, bmAttrCount);
        return metrics;
    }

    public Map<String, Map<String, Set<AtlasEntityHeader>>> getEntitiesByTagDef(String tenant) {
        return Collections.unmodifiableMap(this.tenantTagDefEntitiesMap.getOrDefault(tenant, new HashMap()));
    }

    public Map<String, Map<String, Set<AtlasEntityHeader>>> countTags(String tenant) {
        Map<String, Map<String, Set<AtlasEntityHeader>>> tagDefEntityMap = new HashMap<String, Map<String, Set<AtlasEntityHeader>>>();
        Collection classificationDefNames = this.typeRegistry.getAllClassificationDefNames();
        if (classificationDefNames == null || classificationDefNames.isEmpty()) {
            this.writeTenantTagDefEntitiesMap(tenant, tagDefEntityMap);
            return tagDefEntityMap;
        }
        String tenantPrefix = QualifiedNameGenerator.ensureTypeTenantPrefix(tenant, "");
        Set<String> tenantTagDefNames = classificationDefNames.stream().filter(name -> name.startsWith(tenantPrefix)).collect(Collectors.toSet());
        if (tenantTagDefNames.isEmpty()) {
            this.writeTenantTagDefEntitiesMap(tenant, tagDefEntityMap);
            return tagDefEntityMap;
        }
        if (!this.countTagsTestingEnabled) {
            String method;
            long[] latency = new long[1];
            if (this.countTagsBySearchTenants.contains(tenant) || this.useAtlasSearchToCountTags) {
                tagDefEntityMap = this.countTagsBySearch(tenant, tenantTagDefNames, latency);
                method = "countTagsBySearch";
            } else {
                tagDefEntityMap = this.countTagsByIndexQuery(tenant, tenantTagDefNames, latency);
                method = "countTagsByIndexQuery";
            }
            LOG.info("{} - Updated TagDefEntityMap for {} tag defs, {} took {} ms", new Object[]{tenant, tagDefEntityMap.keySet().size(), method, latency[0]});
            this.writeTenantTagDefEntitiesMap(tenant, tagDefEntityMap);
            return tagDefEntityMap;
        }
        long[] indexQueryLatency = new long[1];
        long[] searchLatency = new long[1];
        CompletableFuture<Map> countByIndexQuery = CompletableFuture.supplyAsync(() -> this.countTagsByIndexQuery(tenant, tenantTagDefNames, indexQueryLatency), this.tagQueryThreadPool);
        CompletableFuture<Map> countBySearch = CompletableFuture.supplyAsync(() -> this.countTagsBySearch(tenant, tenantTagDefNames, searchLatency), this.tagSearchThreadPool);
        CompletableFuture<Void> allDone = CompletableFuture.allOf(countByIndexQuery, countBySearch);
        allDone.join();
        try {
            Map tagDefEntityMapForComparison;
            if (this.useAtlasSearchToCountTags) {
                tagDefEntityMap = countBySearch.get();
                tagDefEntityMapForComparison = countByIndexQuery.get();
            } else {
                tagDefEntityMap = countByIndexQuery.get();
                tagDefEntityMapForComparison = countBySearch.get();
            }
            LOG.info("Updated TagDefEntityMap for {} - countByIndexQuery {} ms, countBySearch {} ms", new Object[]{tenant, indexQueryLatency[0], searchLatency[0]});
            LOG.info("{} - Two tag def entity maps are identical: {}", (Object)tenant, (Object)this.deepCompare(tenant, tagDefEntityMap, tagDefEntityMapForComparison));
        }
        catch (Exception e) {
            LOG.error("Unexpected exception", (Throwable)e);
        }
        this.writeTenantTagDefEntitiesMap(tenant, tagDefEntityMap);
        return tagDefEntityMap;
    }

    @Override
    public void close() {
        if (this.countTagsPersistentCacheEnabled) {
            LOG.info("Persisting tag entity cache before closure");
            this.persistTenantTagDefEntitiesMap();
        }
    }

    private void writeTenantTagDefEntitiesMap(String tenant, Map<String, Map<String, Set<AtlasEntityHeader>>> map) {
        if (this.countTagsPersistentCacheEnabled) {
            this.countTagsPersistentCacheLock.readLock().lock();
            try {
                this.tenantTagDefEntitiesMap.put(tenant, map);
            }
            finally {
                this.countTagsPersistentCacheLock.readLock().unlock();
            }
        } else {
            this.tenantTagDefEntitiesMap.put(tenant, map);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void persistTenantTagDefEntitiesMap() {
        Path tempFilePath = Paths.get(this.cacheDir, "tenant_tag_def_entity.ser.tmp");
        Path finalFilePath = Paths.get(this.cacheDir, TENANT_TAG_DEF_ENTITY_CACHE_FILENAME);
        this.countTagsPersistentCacheLock.writeLock().lock();
        try (ObjectOutputStream oos = new ObjectOutputStream(Files.newOutputStream(tempFilePath, new OpenOption[0]));){
            oos.writeObject(this.tenantTagDefEntitiesMap);
            Files.move(tempFilePath, finalFilePath, StandardCopyOption.REPLACE_EXISTING);
            LOG.info("Tag entity cache persisted");
        }
        catch (Exception e) {
            LOG.error("Failed to persist tag entity cache", (Throwable)e);
        }
        finally {
            this.countTagsPersistentCacheLock.writeLock().unlock();
        }
    }

    private long getTypeCount(String typeName, AtlasEntity.Status status, AtlasStructType.AtlasAttribute tenantAttr, String tenant) {
        Long ret = null;
        String indexQuery = this.indexSearchPrefix + "\"" + Constants.ENTITY_TYPE_PROPERTY_KEY + "\" : (%s)" + " AND " + this.indexSearchPrefix + "\"" + Constants.STATE_PROPERTY_KEY + "\" : (%s)";
        if (tenantAttr != null && tenant != null) {
            indexQuery = indexQuery + " AND " + this.indexSearchPrefix + "\"" + tenantAttr.getVertexPropertyName() + "\" : (%s)";
        }
        indexQuery = String.format(indexQuery, typeName, status.name(), tenant);
        try {
            ret = this.atlasGraph.indexQuery("vertex_index", indexQuery).vertexTotals();
        }
        catch (Exception e) {
            LOG.error("Failed fetching using indexQuery: " + e.getMessage());
        }
        return ret == null ? 0L : ret;
    }

    private boolean deepCompare(String tenant, Map<String, Map<String, Set<AtlasEntityHeader>>> map1, Map<String, Map<String, Set<AtlasEntityHeader>>> map2) {
        String key;
        if (map1 == null || map2 == null) {
            TenantMetricsService.logInconsistency(tenant, "One of the maps is null", map1, map2);
            return false;
        }
        boolean areEqual = true;
        for (Map.Entry<String, Map<String, Set<AtlasEntityHeader>>> entry : map1.entrySet()) {
            key = entry.getKey();
            if (!map2.containsKey(key)) {
                TenantMetricsService.logInconsistency(tenant, "Key missing in second map: " + key, map1.get(key), null);
                areEqual = false;
                continue;
            }
            if (this.deepCompareNestedMap(tenant, key, map1.get(key), map2.get(key))) continue;
            areEqual = false;
        }
        for (Map.Entry<String, Map<String, Set<AtlasEntityHeader>>> entry : map2.entrySet()) {
            key = entry.getKey();
            if (map1.containsKey(key)) continue;
            TenantMetricsService.logInconsistency(tenant, "Key missing in first map: " + key, null, map2.get(key));
            areEqual = false;
        }
        return areEqual;
    }

    private boolean deepCompareNestedMap(String tenant, String key, Map<String, Set<AtlasEntityHeader>> nestedMap1, Map<String, Set<AtlasEntityHeader>> nestedMap2) {
        String nestedKey;
        if (nestedMap1 == null || nestedMap2 == null) {
            TenantMetricsService.logInconsistency(tenant, "One of the nested maps is null for key: " + key, nestedMap1, nestedMap2);
            return false;
        }
        boolean areEqual = nestedMap1.size() == nestedMap2.size();
        for (Map.Entry<String, Set<AtlasEntityHeader>> entry : nestedMap1.entrySet()) {
            nestedKey = entry.getKey();
            if (!nestedMap2.containsKey(nestedKey)) {
                TenantMetricsService.logInconsistency(tenant, "Nested key missing in second map: " + nestedKey, nestedMap1.get(nestedKey), null);
                areEqual = false;
                continue;
            }
            if (this.compareSets(nestedMap1.get(nestedKey), nestedMap2.get(nestedKey))) continue;
            TenantMetricsService.logInconsistency(tenant, "Values differ for nested key: " + nestedKey, nestedMap1.get(nestedKey), nestedMap2.get(nestedKey));
            areEqual = false;
        }
        for (Map.Entry<String, Set<AtlasEntityHeader>> entry : nestedMap2.entrySet()) {
            nestedKey = entry.getKey();
            if (nestedMap1.containsKey(nestedKey)) continue;
            TenantMetricsService.logInconsistency(tenant, "Nested key missing in first map: " + nestedKey, null, nestedMap2.get(nestedKey));
            areEqual = false;
        }
        return areEqual;
    }

    private boolean compareSets(Set<AtlasEntityHeader> set1, Set<AtlasEntityHeader> set2) {
        Set qualifiedNames1 = set1.stream().map(header -> (String)header.getAttribute("qualifiedName")).collect(Collectors.toSet());
        Set qualifiedNames2 = set2.stream().map(header -> (String)header.getAttribute("qualifiedName")).collect(Collectors.toSet());
        return qualifiedNames1.equals(qualifiedNames2);
    }

    private static void logInconsistency(String tenant, String message, Object obj1, Object obj2) {
        LOG.info("{} - Inconsistency found: {}. Map 1: {}, Map 2: {}", new Object[]{tenant, message, obj1, obj2});
    }

    private Map<String, Map<String, Set<AtlasEntityHeader>>> countTagsBySearch(String tenant, Collection<String> classificationDefNames, long[] latency) {
        long start = System.currentTimeMillis();
        HashMap<String, Map<String, Set<AtlasEntityHeader>>> tagDefEntityMap = new HashMap<String, Map<String, Set<AtlasEntityHeader>>>();
        for (String classificationDefName : classificationDefNames) {
            List<AtlasEntityHeader> entities;
            String typeTenantPrefix = CatalogTenantUtils.tenantToEscaped(tenant) + "__";
            String tagDefName = classificationDefName.replace(typeTenantPrefix, "");
            tagDefEntityMap.computeIfAbsent(tagDefName, k -> new HashMap());
            HashSet<String> attributes = new HashSet<String>();
            attributes.add("tenant");
            attributes.add("name");
            attributes.add("qualifiedName");
            try {
                entities = this.searchService.searchAllNonDeprecatedEntities(null, null, tenant, attributes, Collections.singleton(classificationDefName));
            }
            catch (AtlasBaseException e) {
                LOG.error("Failed to count entities by tag", (Throwable)e);
                continue;
            }
            entities.stream().filter(entity -> this.countEntityByName(entity.getTypeName(), (String)entity.getAttribute("qualifiedName"), null)).forEach(entity -> {
                if (this.countTagsTestingEnabled) {
                    LOG.info("{} - Input TagDef: {}, entity: {}", new Object[]{tenant, tagDefName, entity});
                }
                ((Map)tagDefEntityMap.get(tagDefName)).computeIfAbsent(entity.getTypeName(), k -> new HashSet()).add(entity);
            });
        }
        if (latency.length > 0) {
            latency[0] = System.currentTimeMillis() - start;
        }
        return tagDefEntityMap;
    }

    private Map<String, Map<String, Set<AtlasEntityHeader>>> countTagsByIndexQuery(String tenant, Collection<String> classificationDefNames, long[] latency) {
        long start = System.currentTimeMillis();
        HashMap<String, Map<String, Set<AtlasEntityHeader>>> tagDefEntityMap = new HashMap<String, Map<String, Set<AtlasEntityHeader>>>();
        String typeTenantPrefix = CatalogTenantUtils.tenantToEscaped(tenant) + "__";
        for (String classificationDefName : classificationDefNames) {
            Iterator tagIterator;
            String tagDefName = classificationDefName.replace(typeTenantPrefix, "");
            tagDefEntityMap.computeIfAbsent(tagDefName, k -> new HashMap());
            String indexQuery = this.indexSearchPrefix + "\"" + Constants.ENTITY_TYPE_PROPERTY_KEY + "\" : (%s)" + " AND " + this.indexSearchPrefix + "\"" + Constants.STATE_PROPERTY_KEY + "\" : (%s)";
            indexQuery = String.format(indexQuery, classificationDefName, AtlasEntity.Status.ACTIVE.name());
            try {
                tagIterator = this.atlasGraph.indexQuery("vertex_index", indexQuery).vertices();
            }
            catch (Exception e) {
                LOG.error("CountTags failed fetching using indexQuery", (Throwable)e);
                continue;
            }
            while (tagIterator.hasNext()) {
                for (AtlasEdge tagEdge : ((AtlasIndexQuery.Result)tagIterator.next()).getVertex().getEdges(AtlasEdgeDirection.IN)) {
                    AtlasVertex entityVertex = tagEdge.getOutVertex();
                    AtlasEntityHeader entity = null;
                    try {
                        entity = this.entityGraphRetriever.toAtlasEntityHeader(entityVertex);
                    }
                    catch (AtlasBaseException e) {
                        LOG.error("Failed to convert vertex to entity header", (Throwable)e);
                    }
                    if (entity == null) continue;
                    String entityTypeName = entity.getTypeName();
                    String entityQualifiedName = (String)entity.getAttribute("qualifiedName");
                    String qualifiedTagDefName = (String)tagEdge.getInVertex().getProperty("__typeName", String.class);
                    boolean isDeprecated = this.isNonZeroDate(entity, "deprecatedTime");
                    boolean isDeactivated = this.isNonZeroDate(entity, "deactivateTime");
                    if (this.countTagsTestingEnabled) {
                        LOG.info("{} - Input TagDef: {}, entity: {}, deprecated: {}, deactivated: {}", new Object[]{tenant, tagDefName, entity, isDeprecated, isDeactivated});
                    }
                    if (isDeprecated || isDeactivated || !classificationDefName.equals(qualifiedTagDefName) || !this.countEntityByName(entityTypeName, entityQualifiedName, null)) continue;
                    ((Map)tagDefEntityMap.get(tagDefName)).computeIfAbsent(entityTypeName, k -> new HashSet()).add(entity);
                }
            }
        }
        if (latency.length > 0) {
            latency[0] = System.currentTimeMillis() - start;
        }
        return tagDefEntityMap;
    }

    private void countBM(String bmName, String anonymizedBMDefName, AtlasStructType.AtlasAttribute tenantAttr, String tenant, Map<String, Set<String>> countByEntityType, Map<String, Long> bmAttachedEntityCount) {
        Iterator iterator;
        String attributeName = bmName + "." + "__bmName";
        String indexQuery = this.indexSearchPrefix + "\"" + attributeName + "\" : (%s)" + " AND " + this.indexSearchPrefix + "\"" + Constants.STATE_PROPERTY_KEY + "\" : (%s)";
        if (tenantAttr != null && tenant != null) {
            indexQuery = indexQuery + " AND " + this.indexSearchPrefix + "\"" + tenantAttr.getVertexPropertyName() + "\" : (%s)";
        }
        String attributeValue = QualifiedNameGenerator.stripTypeTenantPrefix(tenant, bmName);
        indexQuery = String.format(indexQuery, AtlasStructType.AtlasAttribute.escapeIndexQueryValue((String)attributeValue), AtlasEntity.Status.ACTIVE.name(), tenant);
        try {
            iterator = this.atlasGraph.indexQuery("vertex_index", indexQuery).vertices();
        }
        catch (Exception e) {
            LOG.error("CountBM failed fetching using indexQuery", (Throwable)e);
            return;
        }
        long entityCount = 0L;
        while (iterator.hasNext()) {
            String entityQualifiedName;
            String entityTypeName;
            AtlasVertex entityVertex = ((AtlasIndexQuery.Result)iterator.next()).getVertex();
            AtlasEntityHeader entity = null;
            try {
                entity = this.entityGraphRetriever.toAtlasEntityHeader(entityVertex);
            }
            catch (AtlasBaseException e) {
                LOG.error("Failed to convert vertex to entity header", (Throwable)e);
            }
            if (entity == null || this.isNonZeroDate(entity, "deprecatedTime") || this.isNonZeroDate(entity, "deactivateTime") || !this.countEntityByName(entityTypeName = entity.getTypeName(), entityQualifiedName = (String)entity.getAttribute("qualifiedName"), countByEntityType)) continue;
            ++entityCount;
        }
        if (entityCount > 0L) {
            bmAttachedEntityCount.put(anonymizedBMDefName, entityCount);
        }
    }

    private boolean countEntityByName(String entityTypeName, String qualifiedName, Map<String, Set<String>> countByEntityType) {
        if (entityTypeName == null || qualifiedName == null) {
            LOG.warn("Either typeName or qualifiedName is null, should not have happened.");
            return false;
        }
        if (entityTypeName.startsWith("sr_") && SchemaAtlasTypes.get(entityTypeName) != null || ModelConstants.EntityTypes.get(entityTypeName) != null) {
            if (countByEntityType != null) {
                countByEntityType.computeIfAbsent(entityTypeName, k -> new HashSet()).add(qualifiedName);
            }
        } else {
            LOG.warn("Undefined entity type: {}", (Object)entityTypeName);
            return false;
        }
        return true;
    }

    private boolean isNonZeroDate(AtlasEntityHeader entity, String attribute) {
        Date date = MetadataRegistry.getDateAttribute(entity.getAttribute(attribute));
        return date != null && !ModelConstants.UNIX_ZERO_EPOCH.equals(date);
    }

    private long getTypeShellCount(String typeName, AtlasStructType.AtlasAttribute tenantAttr, String tenant) {
        Long ret = null;
        String indexQuery = this.indexSearchPrefix + "\"" + Constants.ENTITY_TYPE_PROPERTY_KEY + "\" : (%s)" + " AND " + this.indexSearchPrefix + "\"" + Constants.IS_INCOMPLETE_PROPERTY_KEY + "\" : " + Constants.INCOMPLETE_ENTITY_VALUE + " AND " + this.indexSearchPrefix + "\"" + tenantAttr.getVertexPropertyName() + "\" : (%s)";
        indexQuery = String.format(indexQuery, typeName, tenant);
        try {
            ret = this.atlasGraph.indexQuery("vertex_index", indexQuery).vertexTotals();
        }
        catch (Exception e) {
            LOG.error("Failed fetching using indexQuery: " + e.getMessage());
        }
        return ret == null ? 0L : ret;
    }

    private int getAllTypesCount() {
        Collection allTypeNames = this.typeRegistry.getAllTypeNames();
        return CollectionUtils.isNotEmpty((Collection)allTypeNames) ? allTypeNames.size() : 0;
    }

    private Collection<AtlasVertex> getVerticesFromIndexQueryResult(Iterator<AtlasIndexQuery.Result> idxQueryResult, Collection<AtlasVertex> vertices) {
        if (idxQueryResult != null) {
            while (idxQueryResult.hasNext()) {
                AtlasVertex vertex = idxQueryResult.next().getVertex();
                vertices.add(vertex);
            }
        }
        return vertices;
    }

    private String anonymize(String str) {
        if (str == null) {
            return null;
        }
        return this.hashCache.computeIfAbsent(str, TenantMetricsService::computeHash);
    }

    private static String computeHash(String str) {
        try {
            MessageDigest md = MessageDigest.getInstance(HASH_ALGORITHM);
            md.update(str.getBytes(StandardCharsets.UTF_8));
            byte[] strHash = md.digest();
            return new BigInteger(1, strHash).toString(36);
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error("Unsupported message digest algorithm: MD5", (Throwable)e);
            return null;
        }
    }
}

