/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.elasticsearch;

import io.confluent.connect.elasticsearch.ConfigCallbackHandler;
import io.confluent.connect.elasticsearch.ElasticsearchSinkConnectorConfig;
import io.confluent.connect.elasticsearch.Version;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.http.HttpHost;
import org.apache.kafka.common.config.Config;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.ConfigValue;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.MainResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Validator {
    private static final Logger log = LoggerFactory.getLogger(Validator.class);
    private static final String CONNECTOR_V11_COMPATIBLE_ES_VERSION = "7.0.0";
    private static final String DATA_STREAM_COMPATIBLE_ES_VERSION = "7.9.0";
    private ElasticsearchSinkConnectorConfig config;
    private Map<String, ConfigValue> values;
    private List<ConfigValue> validations;
    private ClientFactory clientFactory;

    public Validator(Map<String, String> props) {
        this(props, null);
    }

    protected Validator(Map<String, String> props, ClientFactory clientFactory) {
        try {
            this.config = new ElasticsearchSinkConnectorConfig(props);
        }
        catch (ConfigException configException) {
            // empty catch block
        }
        this.clientFactory = clientFactory == null ? this::createClient : clientFactory;
        this.validations = ElasticsearchSinkConnectorConfig.CONFIG.validate(props);
        this.values = this.validations.stream().collect(Collectors.toMap(ConfigValue::name, Function.identity()));
    }

    public Config validate() {
        if (this.config == null) {
            return new Config(this.validations);
        }
        this.validateCredentials();
        this.validateDataStreamConfigs();
        this.validateIgnoreConfigs();
        this.validateKerberos();
        this.validateLingerMs();
        this.validateMaxBufferedRecords();
        this.validateProxy();
        this.validateSsl();
        if (!this.hasErrors()) {
            try (RestHighLevelClient client = this.clientFactory.client();){
                this.validateConnection(client);
                this.validateVersion(client);
            }
            catch (IOException e) {
                log.warn("Closing the client failed.", (Throwable)e);
            }
            catch (Throwable e) {
                log.error("Failed to create client to verify connection. ", e);
                this.addErrorMessage("connection.url", "Failed to create client to verify connection. " + e.getMessage());
            }
        }
        return new Config(this.validations);
    }

    private void validateCredentials() {
        boolean onlyOneSet = this.config.username() != null ^ this.config.password() != null;
        if (onlyOneSet) {
            String errorMessage = String.format("Both '%s' and '%s' must be set.", "connection.username", "connection.password");
            this.addErrorMessage("connection.username", errorMessage);
            this.addErrorMessage("connection.password", errorMessage);
        }
    }

    private void validateDataStreamConfigs() {
        String errorMessage;
        if (this.config.dataStreamType().toUpperCase().equals(ElasticsearchSinkConnectorConfig.DataStreamType.NONE.name()) ^ this.config.dataStreamDataset().isEmpty()) {
            errorMessage = String.format("Either both or neither '%s' and '%s' must be set.", "data.stream.dataset", "data.stream.type");
            this.addErrorMessage("data.stream.type", errorMessage);
            this.addErrorMessage("data.stream.dataset", errorMessage);
        }
        if (this.config.isDataStream() && this.config.writeMethod() == ElasticsearchSinkConnectorConfig.WriteMethod.UPSERT) {
            errorMessage = String.format("Upserts are not supported with data streams. %s must not be %s if %s and %s are set.", new Object[]{"write.method", ElasticsearchSinkConnectorConfig.WriteMethod.UPSERT, "data.stream.type", "data.stream.dataset"});
            this.addErrorMessage("write.method", errorMessage);
        }
        if (this.config.isDataStream() && this.config.behaviorOnNullValues() == ElasticsearchSinkConnectorConfig.BehaviorOnNullValues.DELETE) {
            errorMessage = String.format("Deletes are not supported with data streams. %s must not be %s if %s and %s are set.", new Object[]{"behavior.on.null.values", ElasticsearchSinkConnectorConfig.BehaviorOnNullValues.DELETE, "data.stream.type", "data.stream.dataset"});
            this.addErrorMessage("behavior.on.null.values", errorMessage);
        }
        if (!this.config.isDataStream() && !this.config.dataStreamTimestampField().isEmpty()) {
            errorMessage = String.format("Mapping a field to the '@timestamp' field is only necessary for data streams. %s must not be set if %s and %s are not set.", "data.stream.timestamp.field", "data.stream.type", "data.stream.dataset");
            this.addErrorMessage("data.stream.timestamp.field", errorMessage);
        }
    }

    private void validateIgnoreConfigs() {
        String errorMessage;
        if (this.config.ignoreKey() && !this.config.ignoreKeyTopics().isEmpty()) {
            errorMessage = String.format("'%s' can not be set if '%s' is true.", "topic.key.ignore", "key.ignore");
            this.addErrorMessage("key.ignore", errorMessage);
            this.addErrorMessage("topic.key.ignore", errorMessage);
        }
        if (this.config.ignoreSchema() && !this.config.ignoreSchemaTopics().isEmpty()) {
            errorMessage = String.format("'%s' can not be set if '%s' is true.", "topic.schema.ignore", "schema.ignore");
            this.addErrorMessage("schema.ignore", errorMessage);
            this.addErrorMessage("topic.schema.ignore", errorMessage);
        }
    }

    private void validateKerberos() {
        String errorMessage;
        boolean onlyOneSet = this.config.kerberosUserPrincipal() != null ^ this.config.keytabPath() != null;
        if (onlyOneSet) {
            errorMessage = String.format("Either both or neither '%s' and '%s' must be set.", "kerberos.user.principal", "kerberos.keytab.path");
            this.addErrorMessage("kerberos.user.principal", errorMessage);
            this.addErrorMessage("kerberos.keytab.path", errorMessage);
        }
        if (this.config.isKerberosEnabled()) {
            if (this.config.isAuthenticatedConnection()) {
                errorMessage = String.format("Either only Kerberos (%s, %s) or connection credentials (%s, %s) must be set.", "kerberos.user.principal", "kerberos.keytab.path", "connection.username", "connection.password");
                this.addErrorMessage("kerberos.user.principal", errorMessage);
                this.addErrorMessage("kerberos.keytab.path", errorMessage);
                this.addErrorMessage("connection.username", errorMessage);
                this.addErrorMessage("connection.password", errorMessage);
            }
            if (this.config.isBasicProxyConfigured()) {
                errorMessage = String.format("Kerberos (%s, %s) is not supported with proxy settings (%s).", "kerberos.user.principal", "kerberos.keytab.path", "proxy.host");
                this.addErrorMessage("kerberos.user.principal", errorMessage);
                this.addErrorMessage("kerberos.keytab.path", errorMessage);
                this.addErrorMessage("proxy.host", errorMessage);
            }
        }
    }

    private void validateLingerMs() {
        if (this.config.lingerMs() > this.config.flushTimeoutMs()) {
            String errorMessage = String.format("'%s' (%d) can not be larger than '%s' (%d).", "linger.ms", this.config.lingerMs(), "flush.timeout.ms", this.config.flushTimeoutMs());
            this.addErrorMessage("linger.ms", errorMessage);
            this.addErrorMessage("flush.timeout.ms", errorMessage);
        }
    }

    private void validateMaxBufferedRecords() {
        if (this.config.maxBufferedRecords() < this.config.batchSize() * this.config.maxInFlightRequests()) {
            String errorMessage = String.format("'%s' (%d) must be larger than or equal to '%s' (%d) x %s (%d).", "max.buffered.records", this.config.maxBufferedRecords(), "batch.size", this.config.batchSize(), "max.in.flight.requests", this.config.maxInFlightRequests());
            this.addErrorMessage("max.buffered.records", errorMessage);
            this.addErrorMessage("batch.size", errorMessage);
            this.addErrorMessage("max.in.flight.requests", errorMessage);
        }
    }

    private void validateProxy() {
        if (!this.config.isBasicProxyConfigured()) {
            String errorMessage;
            if (!this.config.proxyUsername().isEmpty()) {
                errorMessage = String.format("'%s' must be set to use '%s'.", "proxy.host", "proxy.username");
                this.addErrorMessage("proxy.username", errorMessage);
                this.addErrorMessage("proxy.host", errorMessage);
            }
            if (this.config.proxyPassword() != null) {
                errorMessage = String.format("'%s' must be set to use '%s'.", "proxy.host", "proxy.password");
                this.addErrorMessage("proxy.password", errorMessage);
                this.addErrorMessage("proxy.host", errorMessage);
            }
        } else {
            boolean onlyOneSet = this.config.proxyUsername().isEmpty() ^ this.config.proxyPassword() == null;
            if (onlyOneSet) {
                String errorMessage = String.format("Either both or neither '%s' and '%s' can be set.", "proxy.username", "proxy.password");
                this.addErrorMessage("proxy.username", errorMessage);
                this.addErrorMessage("proxy.password", errorMessage);
            }
        }
    }

    private void validateSsl() {
        Map sslConfigs = this.config.originalsWithPrefix("elastic.https.");
        if (!this.config.isSslEnabled()) {
            if (!sslConfigs.isEmpty()) {
                String errorMessage = String.format("'%s' must be set to '%s' to use SSL configs.", new Object[]{"elastic.security.protocol", ElasticsearchSinkConnectorConfig.SecurityProtocol.SSL});
                this.addErrorMessage("elastic.security.protocol", errorMessage);
            }
        } else if (sslConfigs.isEmpty()) {
            String errorMessage = String.format("At least these SSL configs ('%s', '%s', '%s', and '%s') must be present for SSL support. Otherwise set '%s' to '%s'.", new Object[]{"elastic.https.ssl.keystore.location", "elastic.https.ssl.keystore.password", "elastic.https.ssl.truststore.location", "elastic.https.ssl.truststore.password", "elastic.security.protocol", ElasticsearchSinkConnectorConfig.SecurityProtocol.PLAINTEXT});
            this.addErrorMessage("elastic.security.protocol", errorMessage);
        }
    }

    private void validateVersion(RestHighLevelClient client) {
        String errorMessage;
        MainResponse response;
        try {
            response = client.info(RequestOptions.DEFAULT);
        }
        catch (IOException | ElasticsearchStatusException e) {
            return;
        }
        String esVersionNumber = response.getVersion().getNumber();
        if (this.config.isDataStream() && this.compareVersions(esVersionNumber, DATA_STREAM_COMPATIBLE_ES_VERSION) < 0) {
            errorMessage = String.format("Elasticsearch version %s is not compatible with data streams. Elasticsearchversion must be at least %s.", esVersionNumber, DATA_STREAM_COMPATIBLE_ES_VERSION);
            this.addErrorMessage("connection.url", errorMessage);
            this.addErrorMessage("data.stream.type", errorMessage);
            this.addErrorMessage("data.stream.dataset", errorMessage);
        }
        if (this.compareVersions(esVersionNumber, CONNECTOR_V11_COMPATIBLE_ES_VERSION) < 0) {
            errorMessage = String.format("Connector version %s is not compatible with Elasticsearch version %s. Elasticsearch version must be at least %s.", Version.getVersion(), esVersionNumber, CONNECTOR_V11_COMPATIBLE_ES_VERSION);
            this.addErrorMessage("connection.url", errorMessage);
        }
    }

    private int compareVersions(String versionNumber, String compatibleVersion) {
        String[] versionSplit = versionNumber.split("\\.");
        String[] compatibleSplit = compatibleVersion.split("\\.");
        for (int i = 0; i < Math.min(versionSplit.length, compatibleSplit.length); ++i) {
            String versionSplitBeforeSuffix = versionSplit[i].split("-")[0];
            String compatibleSplitBeforeSuffix = compatibleSplit[i].split("-")[0];
            int comparison = Integer.compare(Integer.parseInt(versionSplitBeforeSuffix), Integer.parseInt(compatibleSplitBeforeSuffix));
            if (comparison == 0) continue;
            return comparison;
        }
        return versionSplit.length - compatibleSplit.length;
    }

    private void validateConnection(RestHighLevelClient client) {
        boolean successful;
        String exceptionMessage = "";
        try {
            successful = client.ping(RequestOptions.DEFAULT);
        }
        catch (ElasticsearchStatusException e) {
            switch (e.status()) {
                case FORBIDDEN: {
                    successful = true;
                    break;
                }
                default: {
                    successful = false;
                    exceptionMessage = String.format("Error message: %s", e.getMessage());
                    break;
                }
            }
        }
        catch (Exception e) {
            successful = false;
            exceptionMessage = String.format("Error message: %s", e.getMessage());
        }
        if (!successful) {
            String errorMessage = String.format("Could not connect to Elasticsearch. %s", exceptionMessage);
            this.addErrorMessage("connection.url", errorMessage);
            if (this.config.isAuthenticatedConnection()) {
                errorMessage = String.format("Could not authenticate the user. Check the '%s' and '%s'. %s", "connection.username", "connection.password", exceptionMessage);
                this.addErrorMessage("connection.username", errorMessage);
                this.addErrorMessage("connection.password", errorMessage);
            }
            if (this.config.isSslEnabled()) {
                errorMessage = String.format("Could not connect to Elasticsearch. Check your SSL settings.%s", exceptionMessage);
                this.addErrorMessage("elastic.security.protocol", errorMessage);
            }
            if (this.config.isKerberosEnabled()) {
                errorMessage = String.format("Could not connect to Elasticsearch. Check your Kerberos settings. %s", exceptionMessage);
                this.addErrorMessage("kerberos.user.principal", errorMessage);
                this.addErrorMessage("kerberos.keytab.path", errorMessage);
            }
            if (this.config.isBasicProxyConfigured()) {
                errorMessage = String.format("Could not connect to Elasticsearch. Check your proxy settings. %s", exceptionMessage);
                this.addErrorMessage("proxy.host", errorMessage);
                this.addErrorMessage("proxy.port", errorMessage);
                if (this.config.isProxyWithAuthenticationConfigured()) {
                    this.addErrorMessage("proxy.username", errorMessage);
                    this.addErrorMessage("proxy.password", errorMessage);
                }
            }
        }
    }

    private void addErrorMessage(String property, String error) {
        this.values.get(property).addErrorMessage(error);
    }

    private RestHighLevelClient createClient() {
        ConfigCallbackHandler configCallbackHandler = new ConfigCallbackHandler(this.config);
        return new RestHighLevelClient(RestClient.builder((HttpHost[])this.config.connectionUrls().stream().map(HttpHost::create).collect(Collectors.toList()).toArray(new HttpHost[this.config.connectionUrls().size()])).setHttpClientConfigCallback((RestClientBuilder.HttpClientConfigCallback)configCallbackHandler));
    }

    private boolean hasErrors() {
        for (ConfigValue config : this.validations) {
            if (config.errorMessages().isEmpty()) continue;
            return true;
        }
        return false;
    }

    static interface ClientFactory {
        public RestHighLevelClient client();
    }
}

