/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.source;

import io.confluent.connect.jdbc.source.DataConverter;
import io.confluent.connect.jdbc.source.TableQuerier;
import io.confluent.connect.jdbc.source.TimestampIncrementingOffset;
import io.confluent.connect.jdbc.util.JdbcUtils;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.TimeZone;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.source.SourceRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimestampIncrementingTableQuerier
extends TableQuerier {
    private static final Logger log = LoggerFactory.getLogger(TimestampIncrementingTableQuerier.class);
    private static final Calendar UTC_CALENDAR = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
    private static final BigDecimal LONG_MAX_VALUE_AS_BIGDEC = new BigDecimal(Long.MAX_VALUE);
    private String timestampColumn;
    private String incrementingColumn;
    private long timestampDelay;
    private TimestampIncrementingOffset offset;

    public TimestampIncrementingTableQuerier(TableQuerier.QueryMode mode, String name, String topicPrefix, String timestampColumn, String incrementingColumn, Map<String, Object> offsetMap, Long timestampDelay, String schemaPattern) {
        super(mode, name, topicPrefix, schemaPattern);
        this.timestampColumn = timestampColumn;
        this.incrementingColumn = incrementingColumn;
        this.timestampDelay = timestampDelay;
        this.offset = TimestampIncrementingOffset.fromMap(offsetMap);
    }

    @Override
    protected void createPreparedStatement(Connection db) throws SQLException {
        if (this.incrementingColumn != null && this.incrementingColumn.isEmpty()) {
            this.incrementingColumn = JdbcUtils.getAutoincrementColumn(db, this.schemaPattern, this.name);
        }
        String quoteString = JdbcUtils.getIdentifierQuoteString(db);
        StringBuilder builder = new StringBuilder();
        switch (this.mode) {
            case TABLE: {
                builder.append("SELECT * FROM ");
                builder.append(JdbcUtils.quoteString(this.name, quoteString));
                break;
            }
            case QUERY: {
                builder.append(this.query);
                break;
            }
            default: {
                throw new ConnectException("Unknown mode encountered when preparing query: " + this.mode.toString());
            }
        }
        if (this.incrementingColumn != null && this.timestampColumn != null) {
            builder.append(" WHERE ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" < ? AND ((");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" = ? AND ");
            builder.append(JdbcUtils.quoteString(this.incrementingColumn, quoteString));
            builder.append(" > ?");
            builder.append(") OR ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" > ?)");
            builder.append(" ORDER BY ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(",");
            builder.append(JdbcUtils.quoteString(this.incrementingColumn, quoteString));
            builder.append(" ASC");
        } else if (this.incrementingColumn != null) {
            builder.append(" WHERE ");
            builder.append(JdbcUtils.quoteString(this.incrementingColumn, quoteString));
            builder.append(" > ?");
            builder.append(" ORDER BY ");
            builder.append(JdbcUtils.quoteString(this.incrementingColumn, quoteString));
            builder.append(" ASC");
        } else if (this.timestampColumn != null) {
            builder.append(" WHERE ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" > ? AND ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" < ? ORDER BY ");
            builder.append(JdbcUtils.quoteString(this.timestampColumn, quoteString));
            builder.append(" ASC");
        }
        String queryString = builder.toString();
        log.debug("{} prepared SQL query: {}", (Object)this, (Object)queryString);
        this.stmt = db.prepareStatement(queryString);
    }

    @Override
    protected ResultSet executeQuery() throws SQLException {
        if (this.incrementingColumn != null && this.timestampColumn != null) {
            Timestamp tsOffset = this.offset.getTimestampOffset();
            Long incOffset = this.offset.getIncrementingOffset();
            Timestamp endTime = new Timestamp(JdbcUtils.getCurrentTimeOnDB(this.stmt.getConnection(), UTC_CALENDAR).getTime() - this.timestampDelay);
            this.stmt.setTimestamp(1, endTime, UTC_CALENDAR);
            this.stmt.setTimestamp(2, tsOffset, UTC_CALENDAR);
            this.stmt.setLong(3, incOffset);
            this.stmt.setTimestamp(4, tsOffset, UTC_CALENDAR);
            log.debug("Executing prepared statement with start time value = {} end time = {} and incrementing value = {}", new Object[]{JdbcUtils.formatUTC(tsOffset), JdbcUtils.formatUTC(endTime), incOffset});
        } else if (this.incrementingColumn != null) {
            Long incOffset = this.offset.getIncrementingOffset();
            this.stmt.setLong(1, incOffset);
            log.debug("Executing prepared statement with incrementing value = {}", (Object)incOffset);
        } else if (this.timestampColumn != null) {
            Timestamp tsOffset = this.offset.getTimestampOffset();
            Timestamp endTime = new Timestamp(JdbcUtils.getCurrentTimeOnDB(this.stmt.getConnection(), UTC_CALENDAR).getTime() - this.timestampDelay);
            this.stmt.setTimestamp(1, tsOffset, UTC_CALENDAR);
            this.stmt.setTimestamp(2, endTime, UTC_CALENDAR);
            log.debug("Executing prepared statement with timestamp value = {} end time = {}", (Object)JdbcUtils.formatUTC(tsOffset), (Object)JdbcUtils.formatUTC(endTime));
        }
        return this.stmt.executeQuery();
    }

    @Override
    public SourceRecord extractRecord() throws SQLException {
        String topic;
        Map<String, String> partition;
        Struct record = DataConverter.convertRecord(this.schema, this.resultSet);
        this.offset = this.extractOffset(this.schema, record);
        switch (this.mode) {
            case TABLE: {
                partition = Collections.singletonMap("table", this.name);
                topic = this.topicPrefix + this.name;
                break;
            }
            case QUERY: {
                partition = Collections.singletonMap("query", "query");
                topic = this.topicPrefix;
                break;
            }
            default: {
                throw new ConnectException("Unexpected query mode: " + (Object)((Object)this.mode));
            }
        }
        return new SourceRecord(partition, this.offset.toMap(), topic, record.schema(), (Object)record);
    }

    TimestampIncrementingOffset extractOffset(Schema schema, Struct record) {
        Long extractedId;
        Timestamp extractedTimestamp;
        if (this.timestampColumn != null) {
            extractedTimestamp = (Timestamp)record.get(this.timestampColumn);
            Timestamp timestampOffset = this.offset.getTimestampOffset();
            assert (timestampOffset != null && timestampOffset.compareTo(extractedTimestamp) <= 0);
        } else {
            extractedTimestamp = null;
        }
        if (this.incrementingColumn != null) {
            Schema incrementingColumnSchema = schema.field(this.incrementingColumn).schema();
            Object incrementingColumnValue = record.get(this.incrementingColumn);
            if (incrementingColumnValue == null) {
                throw new ConnectException("Null value for incrementing column of type: " + incrementingColumnSchema.type());
            }
            if (this.isIntegralPrimitiveType(incrementingColumnValue)) {
                extractedId = ((Number)incrementingColumnValue).longValue();
            } else if (incrementingColumnSchema.name() != null && incrementingColumnSchema.name().equals("org.apache.kafka.connect.data.Decimal")) {
                BigDecimal decimal = (BigDecimal)incrementingColumnValue;
                if (decimal.compareTo(LONG_MAX_VALUE_AS_BIGDEC) > 0) {
                    throw new ConnectException("Decimal value for incrementing column exceeded Long.MAX_VALUE");
                }
                if (decimal.scale() != 0) {
                    throw new ConnectException("Scale of Decimal value for incrementing column must be 0");
                }
                extractedId = decimal.longValue();
            } else {
                throw new ConnectException("Invalid type for incrementing column: " + incrementingColumnSchema.type());
            }
            Long incrementingOffset = this.offset.getIncrementingOffset();
            assert (incrementingOffset == -1L || extractedId > incrementingOffset || this.timestampColumn != null);
        } else {
            extractedId = null;
        }
        return new TimestampIncrementingOffset(extractedTimestamp, extractedId);
    }

    private boolean isIntegralPrimitiveType(Object incrementingColumnValue) {
        return incrementingColumnValue instanceof Long || incrementingColumnValue instanceof Integer || incrementingColumnValue instanceof Short || incrementingColumnValue instanceof Byte;
    }

    public String toString() {
        return "TimestampIncrementingTableQuerier{name='" + this.name + '\'' + ", query='" + this.query + '\'' + ", topicPrefix='" + this.topicPrefix + '\'' + ", timestampColumn='" + this.timestampColumn + '\'' + ", incrementingColumn='" + this.incrementingColumn + '\'' + '}';
    }
}

