/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.tools;

import com.google.common.collect.ImmutableList;
import io.confluent.kafka.schemaregistry.ParsedSchema;
import io.confluent.kafka.schemaregistry.avro.AvroSchemaProvider;
import io.confluent.kafka.schemaregistry.client.SchemaMetadata;
import io.confluent.kafka.schemaregistry.client.SchemaRegistryClient;
import io.confluent.kafka.schemaregistry.client.SchemaRegistryClientFactory;
import io.confluent.kafka.schemaregistry.client.rest.entities.Schema;
import io.confluent.kafka.schemaregistry.client.rest.exceptions.RestClientException;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.Callable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;

@CommandLine.Command(name="check-schema-compatibility", mixinStandardHelpOptions=true, description={"Compare subjects between two Schema Registry instances and check compatibility. Requires --config to specify a configuration file with both source and target Schema Registry connection details using prefixed properties."}, sortOptions=false, sortSynopsis=false)
public class CheckSchemaCompatibility
implements Callable<Integer> {
    private static final Logger LOG = LoggerFactory.getLogger(CheckSchemaCompatibility.class);
    private static final String SCHEMA_REGISTRY_URL_CONFIG = "schema.registry.url";
    @CommandLine.Option(names={"--config", "-c"}, description={"Path to configuration file containing both source and target Schema Registry connection details. Use prefixed properties like 'source.schema.registry.url' and 'target.schema.registry.url'.\nExample config file:\nsource.schema.registry.url=<sr-url>\nsource.context=:.custom_context\nsource.credential=apikey:secret\ntarget.schema.registry.url=<sr-url>\ntarget.context=:.custom_context\ntarget.credential=apikey:secret"}, paramLabel="<file>", required=true)
    private String configFile;
    @CommandLine.Option(names={"--verbose", "-v"}, description={"Enable verbose output to show detailed subject and version information"})
    private boolean verbose = false;
    private String sourceUrl;
    private String targetUrl;
    private String sourceContext = ":.:";
    private String targetContext = ":.:";
    private String sourceCredential;
    private String targetCredential;

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Integer call() throws Exception {
        try {
            this.loadConfigurations();
            LOG.info("Starting schema compatibility check between:");
            LOG.info("  Source: {} (context: {})", (Object)this.sourceUrl, (Object)this.sourceContext);
            LOG.info("  Target: {} (context: {})", (Object)this.targetUrl, (Object)this.targetContext);
            try (SchemaRegistryClient sourceClient = this.createClient(this.sourceUrl, this.sourceCredential);
                 SchemaRegistryClient targetClient = this.createClient(this.targetUrl, this.targetCredential);){
                List<String> sourceSubjects = this.getSubjects(sourceClient, this.sourceUrl, this.sourceContext);
                List<String> targetSubjects = this.getSubjects(targetClient, this.targetUrl, this.targetContext);
                if (!this.compareSubjects(sourceSubjects, targetSubjects, sourceClient, targetClient)) {
                    Integer n = 1;
                    return n;
                }
                LOG.info("Schema compatibility check completed successfully");
                return 0;
            }
        }
        catch (Exception e) {
            LOG.error("Error during schema compatibility check: {}", (Object)e.getMessage(), (Object)e);
            return 1;
        }
    }

    private void loadConfigurations() throws IOException {
        LOG.info("Loading configuration from: {}", (Object)this.configFile);
        Properties allProps = this.loadConfigFile(this.configFile);
        this.sourceUrl = this.getRequiredProperty(allProps, "source.schema.registry.url", "configuration file");
        this.sourceContext = allProps.getProperty("source.context", ":.:");
        this.sourceCredential = allProps.getProperty("source.credential");
        this.targetUrl = this.getRequiredProperty(allProps, "target.schema.registry.url", "configuration file");
        this.targetContext = allProps.getProperty("target.context", ":.:");
        this.targetCredential = allProps.getProperty("target.credential");
        LOG.info("Source configuration loaded - URL: {}, Context: {}, Has credentials: {}", new Object[]{this.sourceUrl, this.sourceContext, this.sourceCredential != null});
        LOG.info("Target configuration loaded - URL: {}, Context: {}, Has credentials: {}", new Object[]{this.targetUrl, this.targetContext, this.targetCredential != null});
    }

    private Properties loadConfigFile(String configFile) throws IOException {
        if (!Files.exists(Paths.get(configFile, new String[0]), new LinkOption[0])) {
            throw new IOException("Configuration file not found: " + configFile);
        }
        Properties props = new Properties();
        try (FileInputStream fis = new FileInputStream(configFile);){
            props.load(fis);
        }
        return props;
    }

    private String getRequiredProperty(Properties props, String key, String configFile) throws IOException {
        String value = props.getProperty(key);
        if (value == null || value.trim().isEmpty()) {
            throw new IOException("Required property '" + key + "' not found in " + configFile);
        }
        return value.trim();
    }

    private SchemaRegistryClient createClient(String url, String credential) {
        HashMap<String, String> clientConfigs = new HashMap<String, String>();
        clientConfigs.put(SCHEMA_REGISTRY_URL_CONFIG, url);
        if (credential != null && !credential.trim().isEmpty()) {
            clientConfigs.put("basic.auth.credentials.source", "USER_INFO");
            clientConfigs.put("basic.auth.user.info", credential);
        }
        return SchemaRegistryClientFactory.newClient(Collections.singletonList(url), (int)1000, (List)ImmutableList.of((Object)new AvroSchemaProvider()), clientConfigs, Collections.emptyMap());
    }

    private List<String> getSubjects(SchemaRegistryClient client, String url, String contextName) throws IOException, RestClientException {
        Collection subjectsCollection = client.getAllSubjectsByPrefix(contextName);
        ArrayList<String> subjects = new ArrayList<String>(subjectsCollection);
        LOG.info("Found {} subjects in {} {} context", new Object[]{subjects.size(), url, contextName});
        if (this.verbose) {
            for (String subject : subjects) {
                LOG.info("  {} subject: {}", (Object)contextName, (Object)subject);
            }
        }
        return subjects;
    }

    protected boolean compareSubjects(List<String> sourceSubjects, List<String> targetSubjects, SchemaRegistryClient sourceClient, SchemaRegistryClient targetClient) throws RestClientException, IOException {
        HashSet<String> sourceSet = new HashSet<String>(sourceSubjects);
        HashSet<String> targetSet = new HashSet<String>(targetSubjects);
        HashSet<String> onlyInSource = new HashSet<String>(sourceSet);
        onlyInSource.removeAll(targetSet);
        HashSet<String> onlyInTarget = new HashSet<String>(targetSet);
        onlyInTarget.removeAll(sourceSet);
        HashSet<String> inBoth = new HashSet<String>(sourceSet);
        inBoth.retainAll(targetSet);
        LOG.info("\n=== COMPARISON RESULTS ===");
        LOG.info("Total subjects in source ({}): {}", (Object)this.sourceContext, (Object)sourceSubjects.size());
        LOG.info("Total subjects in target ({}): {}", (Object)this.targetContext, (Object)targetSubjects.size());
        LOG.info("Subjects in both registries: {}", (Object)inBoth.size());
        LOG.info("Subjects only in source: {}", (Object)onlyInSource.size());
        LOG.info("Subjects only in target: {}", (Object)onlyInTarget.size());
        if (!onlyInSource.isEmpty() && this.verbose) {
            LOG.info("\nSubjects only in source ({}):", (Object)this.sourceContext);
            ArrayList<String> sortedOnlyInSource = new ArrayList<String>(onlyInSource);
            Collections.sort(sortedOnlyInSource);
            for (String subject : sortedOnlyInSource) {
                LOG.info("  - {}", (Object)subject);
            }
        }
        if (!onlyInTarget.isEmpty() && this.verbose) {
            LOG.info("\nSubjects only in target ({}):", (Object)this.targetContext);
            ArrayList<String> sortedOnlyInTarget = new ArrayList<String>(onlyInTarget);
            Collections.sort(sortedOnlyInTarget);
            for (String subject : sortedOnlyInTarget) {
                LOG.info("  - {}", (Object)subject);
            }
        }
        if (sourceSet.isEmpty() || targetSet.isEmpty()) {
            LOG.info("\n\u2713 Compatible, one schema registry is empty");
            return true;
        }
        if (!onlyInTarget.isEmpty()) {
            LOG.error("\n\u2717 Not compatible, target has subjects that don't exist in source: {}", onlyInTarget);
            return false;
        }
        if (!onlyInSource.isEmpty()) {
            LOG.info("\n\u2139 Source has additional subjects (acceptable): {}", onlyInSource);
        }
        if (!inBoth.isEmpty()) {
            ArrayList<String> sortedInBoth = new ArrayList<String>(inBoth);
            Collections.sort(sortedInBoth);
            if (this.verbose) {
                LOG.info("\nSubjects in both registries: {}", sortedInBoth);
            }
            for (String subject : sortedInBoth) {
                if (this.compareSubject(sourceClient, targetClient, subject)) continue;
                return false;
            }
        }
        return true;
    }

    protected boolean compareSubject(SchemaRegistryClient sourceClient, SchemaRegistryClient targetClient, String subject) throws RestClientException, IOException {
        try {
            LOG.info("\n--- Comparing subject: {} ---", (Object)subject);
            List sourceVersions = sourceClient.getAllVersions(subject);
            List targetVersions = targetClient.getAllVersions(subject);
            if (this.verbose) {
                LOG.info("Source versions: {}", (Object)sourceVersions);
                LOG.info("Target versions: {}", (Object)targetVersions);
            }
            HashSet sourceVersionSet = new HashSet(sourceVersions);
            HashSet targetVersionSet = new HashSet(targetVersions);
            HashSet commonVersions = new HashSet(sourceVersionSet);
            commonVersions.retainAll(targetVersionSet);
            HashSet onlyInSource = new HashSet(sourceVersionSet);
            onlyInSource.removeAll(targetVersionSet);
            HashSet onlyInTarget = new HashSet(targetVersionSet);
            onlyInTarget.removeAll(sourceVersionSet);
            if (!onlyInSource.isEmpty() && this.verbose) {
                LOG.info("\u2139 Versions only in source (will be ignored): {}", onlyInSource);
            }
            if (!onlyInTarget.isEmpty()) {
                LOG.error("\u2717 Target has versions that don't exist in source for subject '{}': {}", (Object)subject, onlyInTarget);
                return false;
            }
            LOG.info("Comparing {} common versions for subject '{}'", (Object)commonVersions.size(), (Object)subject);
            boolean allVersionsMatch = true;
            int mismatchCount = 0;
            ArrayList sortedCommonVersions = new ArrayList(commonVersions);
            Collections.sort(sortedCommonVersions);
            for (Integer version : sortedCommonVersions) {
                if (this.compareVersion(sourceClient, targetClient, subject, version)) continue;
                ++mismatchCount;
                allVersionsMatch = false;
            }
            if (allVersionsMatch) {
                LOG.info("\u2713 All {} common versions match perfectly for subject '{}'", (Object)commonVersions.size(), (Object)subject);
                return true;
            }
            LOG.error("\u2717 Subject '{}' has {} mismatched version(s) out of {} common versions", new Object[]{subject, mismatchCount, commonVersions.size()});
            return false;
        }
        catch (Exception e) {
            LOG.error("\u2717 Error comparing subject '{}': {}", (Object)subject, (Object)e.getMessage());
            LOG.error("  Exception details: ", (Throwable)e);
            throw e;
        }
    }

    protected boolean compareVersion(SchemaRegistryClient sourceClient, SchemaRegistryClient targetClient, String subject, Integer version) throws RestClientException, IOException {
        try {
            LOG.info("  Comparing version {} for subject '{}'", (Object)version, (Object)subject);
            SchemaMetadata sourceSchema = sourceClient.getSchemaMetadata(subject, version.intValue());
            SchemaMetadata targetSchema = targetClient.getSchemaMetadata(subject, version.intValue());
            Schema sourceSchemaEntity = new Schema(subject, version, Integer.valueOf(sourceSchema.getId()), sourceSchema.getSchemaType(), sourceSchema.getReferences(), sourceSchema.getMetadata(), sourceSchema.getRuleSet(), sourceSchema.getSchema());
            Schema targetSchemaEntity = new Schema(subject, version, Integer.valueOf(targetSchema.getId()), targetSchema.getSchemaType(), targetSchema.getReferences(), targetSchema.getMetadata(), targetSchema.getRuleSet(), targetSchema.getSchema());
            Optional sourceParsed = sourceClient.parseSchema(sourceSchemaEntity);
            Optional targetParsed = targetClient.parseSchema(targetSchemaEntity);
            if (!sourceParsed.isPresent()) {
                LOG.error("\u2717 Failed to parse source schema for subject '{}' version {}", (Object)subject, (Object)version);
                return false;
            }
            if (!targetParsed.isPresent()) {
                LOG.error("\u2717 Failed to parse target schema for subject '{}' version {}", (Object)subject, (Object)version);
                return false;
            }
            if (!((ParsedSchema)sourceParsed.get()).equivalent((ParsedSchema)targetParsed.get())) {
                LOG.error("\u2717 Schema content mismatch for subject '{}' version {}", (Object)subject, (Object)version);
                LOG.error("  Source schema: {}", (Object)sourceSchema.getSchema());
                LOG.error("  Target schema: {}", (Object)targetSchema.getSchema());
                return false;
            }
            if (!version.equals(sourceSchema.getVersion()) || !version.equals(targetSchema.getVersion())) {
                LOG.error("\u2717 Version number mismatch for subject '{}' version {}: source={}, target={}", new Object[]{subject, version, sourceSchema.getVersion(), targetSchema.getVersion()});
                return false;
            }
            if (sourceSchema.getId() != targetSchema.getId()) {
                LOG.error("\u2139 Schema ID differs for subject '{}' version {}: source={}, target={}", new Object[]{subject, version, sourceSchema.getId(), targetSchema.getId()});
                return false;
            }
            LOG.info("\u2713 Version {} matches for subject '{}'", (Object)version, (Object)subject);
            return true;
        }
        catch (Exception e) {
            LOG.error("\u2717 Error comparing version {} for subject '{}': {}", new Object[]{version, subject, e.getMessage()});
            LOG.error("  Exception details: ", (Throwable)e);
            throw e;
        }
    }

    public static void main(String[] args) {
        CommandLine commandLine = new CommandLine((Object)new CheckSchemaCompatibility());
        commandLine.setUsageHelpLongOptionsMaxWidth(30);
        int exitCode = commandLine.execute(args);
        System.exit(exitCode);
    }
}

