/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.axle.core.streams;

import java.util.Map;
import org.reactivestreams.Publisher;
import java.util.function.Consumer;
import java.util.concurrent.CompletionStage;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a stream of items that can be read from.
 * <p>
 * Any class that implements this interface can be used by a {@link io.vertx.axle.core.streams.Pump} to pump data from it
 * to a {@link io.vertx.axle.core.streams.WriteStream}.
 * <p>
 * <h3>Streaming mode</h3>
 * The stream is either in <i>flowing</i> or <i>fetch</i> mode.
 * <ul>
 *   <i>Initially the stream is in <i>flowing</i> mode.</i>
 *   <li>When the stream is in <i>flowing</i> mode, elements are delivered to the <code>handler</code>.</li>
 *   <li>When the stream is in <i>fetch</i> mode, only the number of requested elements will be delivered to the <code>handler</code>.</li>
 * </ul>
 * The mode can be changed with the {@link io.vertx.axle.core.streams.ReadStream#pause}, {@link io.vertx.axle.core.streams.ReadStream#resume} and {@link io.vertx.axle.core.streams.ReadStream#fetch} methods:
 * <ul>
 *   <li>Calling {@link io.vertx.axle.core.streams.ReadStream#resume} sets the <i>flowing</i> mode</li>
 *   <li>Calling {@link io.vertx.axle.core.streams.ReadStream#pause} sets the <i>fetch</i> mode and resets the demand to <code>0</code></li>
 *   <li>Calling {@link io.vertx.axle.core.streams.ReadStream#fetch} requests a specific amount of elements and adds it to the actual demand</li>
 * </ul>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.streams.ReadStream original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.axle.Gen(io.vertx.core.streams.ReadStream.class)
public interface ReadStream<T> extends io.vertx.axle.core.streams.StreamBase {

  io.vertx.core.streams.ReadStream getDelegate();

  /**
   * Set an exception handler on the read stream.
   * @param handler the exception handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> exceptionHandler(Consumer<Throwable> handler);
  /**
   * Set a data handler. As data is read, the handler will be called with the data.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> handler(Consumer<T> handler);
  /**
   * Pause the <code>ReadStream</code>, it sets the buffer in <code>fetch</code> mode and clears the actual demand.
   * <p>
   * While it's paused, no data will be sent to the data <code>handler</code>.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> pause();
  /**
   * Resume reading, and sets the buffer in <code>flowing</code> mode.
   * <p/>
   * If the <code>ReadStream</code> has been paused, reading will recommence on it.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> resume();
  /**
   * Fetch the specified <code>amount</code> of elements. If the <code>ReadStream</code> has been paused, reading will
   * recommence with the specified <code>amount</code> of items, otherwise the specified <code>amount</code> will
   * be added to the current stream demand.
   * @param amount 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> fetch(long amount);
  /**
   * Set an end handler. Once the stream has ended, and there is no more data to be read, this handler will be called.
   * @param endHandler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> endHandler(Consumer<Void> endHandler);
  /**
   * Pause this stream and return a {@link io.vertx.axle.core.streams.Pipe} to transfer the elements of this stream to a destination {@link io.vertx.axle.core.streams.WriteStream}.
   * <p/>
   * The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.
   * @return a pipe
   */
  public io.vertx.axle.core.streams.Pipe<T> pipe();
  /**
   * Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   * <p>
   * Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   * <p>
   * Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   * called with the result.
   * @param dst the destination write stream
   * @return 
   */
  public CompletionStage<Void> pipeTo(io.vertx.axle.core.streams.WriteStream<T> dst);
  org.reactivestreams.Publisher<T> toPublisher();


  public static <T>ReadStream<T> newInstance(io.vertx.core.streams.ReadStream arg) {
    return arg != null ? new ReadStreamImpl<T>(arg) : null;
  }

  public static <T>ReadStream<T> newInstance(io.vertx.core.streams.ReadStream arg, io.vertx.lang.axle.TypeArg<T> __typeArg_T) {
    return arg != null ? new ReadStreamImpl<T>(arg, __typeArg_T) : null;
  }
}

class ReadStreamImpl<T> implements ReadStream<T> {
  private final io.vertx.core.streams.ReadStream<T> delegate;
  public final io.vertx.lang.axle.TypeArg<T> __typeArg_0;
  
  public ReadStreamImpl(io.vertx.core.streams.ReadStream delegate) {
    this.delegate = delegate;
    this.__typeArg_0 = io.vertx.lang.axle.TypeArg.unknown();  }

  public ReadStreamImpl(io.vertx.core.streams.ReadStream delegate, io.vertx.lang.axle.TypeArg<T> typeArg_0) {
    this.delegate = delegate;
    this.__typeArg_0 = typeArg_0;
  }

  /**
  * Empty constructor used by CDI, do not use this constructor directly.
  **/
  ReadStreamImpl() {    this.delegate = null;
    this.__typeArg_0 = io.vertx.lang.axle.TypeArg.unknown();  }

  public io.vertx.core.streams.ReadStream getDelegate() {
    return delegate;
  }

  private org.reactivestreams.Publisher<T> publisher;

  public synchronized org.reactivestreams.Publisher<T> toPublisher() {
    if (publisher == null) {
      java.util.function.Function<T, T> conv = (java.util.function.Function<T, T>) __typeArg_0.wrap;
      publisher = io.vertx.axle.PublisherHelper.toPublisher(delegate, conv);
    }
    return publisher;
  }

  public synchronized org.eclipse.microprofile.reactive.streams.operators.PublisherBuilder<T> toPublisherBuilder() {
    if (publisher == null) {
      java.util.function.Function<T, T> conv = (java.util.function.Function<T, T>) __typeArg_0.wrap;
      publisher = io.vertx.axle.PublisherHelper.toPublisher(delegate, conv);
    }
    return org.eclipse.microprofile.reactive.streams.operators.ReactiveStreams.fromPublisher(publisher);
  }

  /**
   * Set an exception handler on the read stream.
   * @param handler the exception handler
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.streams.ReadStream<T> __exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  /**
   * Set an exception handler on the read stream.
   * @param handler the exception handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> exceptionHandler(Consumer<Throwable> handler) {
    return __exceptionHandler(handler != null ? handler::accept : null);
  }

  /**
   * Set a data handler. As data is read, the handler will be called with the data.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.streams.ReadStream<T> __handler(Handler<T> handler) { 
    delegate.handler(new Handler<T>() {
      public void handle(T event) {
        handler.handle((T)__typeArg_0.wrap(event));
      }
    });
    return this;
  }

  /**
   * Set a data handler. As data is read, the handler will be called with the data.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> handler(Consumer<T> handler) {
    return __handler(handler != null ? handler::accept : null);
  }

  /**
   * Pause the <code>ReadStream</code>, it sets the buffer in <code>fetch</code> mode and clears the actual demand.
   * <p>
   * While it's paused, no data will be sent to the data <code>handler</code>.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> pause() { 
    delegate.pause();
    return this;
  }

  /**
   * Resume reading, and sets the buffer in <code>flowing</code> mode.
   * <p/>
   * If the <code>ReadStream</code> has been paused, reading will recommence on it.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> resume() { 
    delegate.resume();
    return this;
  }

  /**
   * Fetch the specified <code>amount</code> of elements. If the <code>ReadStream</code> has been paused, reading will
   * recommence with the specified <code>amount</code> of items, otherwise the specified <code>amount</code> will
   * be added to the current stream demand.
   * @param amount 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> fetch(long amount) { 
    delegate.fetch(amount);
    return this;
  }

  /**
   * Set an end handler. Once the stream has ended, and there is no more data to be read, this handler will be called.
   * @param endHandler 
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.streams.ReadStream<T> __endHandler(Handler<Void> endHandler) { 
    delegate.endHandler(endHandler);
    return this;
  }

  /**
   * Set an end handler. Once the stream has ended, and there is no more data to be read, this handler will be called.
   * @param endHandler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.streams.ReadStream<T> endHandler(Consumer<Void> endHandler) {
    return __endHandler(endHandler != null ? endHandler::accept : null);
  }

  /**
   * Pause this stream and return a {@link io.vertx.axle.core.streams.Pipe} to transfer the elements of this stream to a destination {@link io.vertx.axle.core.streams.WriteStream}.
   * <p/>
   * The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.
   * @return a pipe
   */
  public io.vertx.axle.core.streams.Pipe<T> pipe() { 
    io.vertx.axle.core.streams.Pipe<T> ret = io.vertx.axle.core.streams.Pipe.newInstance(delegate.pipe(), __typeArg_0);
    return ret;
  }

  /**
   * Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   * <p>
   * Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   * <p>
   * Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   * called with the result.
   * @param dst the destination write stream
   * @param handler 
   */
  private void __pipeTo(io.vertx.axle.core.streams.WriteStream<T> dst, Handler<AsyncResult<Void>> handler) { 
    delegate.pipeTo(dst.getDelegate(), handler);
  }

  /**
   * Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   * <p>
   * Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   * <p>
   * Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   * called with the result.
   * @param dst the destination write stream
   * @return 
   */
  public CompletionStage<Void> pipeTo(io.vertx.axle.core.streams.WriteStream<T> dst) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __pipeTo(dst, handler);
    });
  }

}
