package org.apache.maven.plugin.multichanges.util;

/* ====================================================================
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements.  See the NOTICE file distributed with
 *   this work for additional information regarding copyright ownership.
 *   The ASF licenses this file to You under the Apache License, Version 2.0
 *   (the "License"); you may not use this file except in compliance with
 *   the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * ====================================================================
 */


import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * This class provides a method to convert a date in the changes plugin format
 * (yyyy-MM-dd) to a date in RSS format (RFC 822).
 * 
 */
public class DateFormatter {
	/**
	 * The date format used in changes report file (changes.xml used in
	 * maven-changes-plugin).
	 */
	private final static SimpleDateFormat changesSDF = new SimpleDateFormat(
			"yyyy-MM-dd");

	/**
	 * The date format defined in RFC 822 and used in RSS feeds.
	 */
	private final static SimpleDateFormat rfc822DF = new SimpleDateFormat(
			"EEE, dd MMM yyyy HH:mm:ss Z", Locale.ENGLISH);

	/**
	 * Log for debug output
	 */
	private static Log log = LogFactory.getLog(DateFormatter.class);

	/**
	 * Converts a date in the changes plugin format (yyyy-MM-dd) to a date in
	 * RSS format (RFC 822).
	 * 
	 * @param changesDate
	 *            the date to convert
	 * @return the date in the RFC 822 format. An empty String if an exception
	 *         occurs.
	 */
	public static final String convertDate(final String changesDate) {
		try {
			return rfc822DF.format(changesSDF.parse(changesDate));
		} catch (ParseException e) {
			if (log.isDebugEnabled())
				log
						.error("Unable to convert the date [" + changesDate
								+ "]", e);
			else
				log.error("Unable to convert the date [" + changesDate + "] : "
						+ e.getMessage());
			return "";
		}
	}

	/**
	 * Test if a String is a date in the changes plugin format (yyyy-MM-dd).
	 * 
	 * @param aString
	 *            the string to test
	 * @return true if the String can be parsed as a Date in the format
	 *         yyyy-MM-dd.
	 */
	public static final boolean isChangesDate(final String aString) {
		try {
			changesSDF.parse(aString);
			return true;
		} catch (ParseException e) {
			return false;
		}
	}
}
