/*
 * Copyright 2016 higherfrequencytrading.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.chronicle.bytes;

import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.Memory;
import net.openhft.chronicle.core.OS;
import net.openhft.chronicle.core.UnsafeMemory;
import net.openhft.chronicle.core.io.Closeable;
import net.openhft.chronicle.core.io.IORuntimeException;
import net.openhft.chronicle.core.util.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;

import static net.openhft.chronicle.core.util.StringUtils.extractChars;

/**
 * Bytes to wrap memory mapped data.
 */
public class MappedBytes extends AbstractBytes<Void> implements Closeable {
    @NotNull
    private final MappedFile mappedFile;

    // assume the mapped file is reserved already.
    protected MappedBytes(@NotNull MappedFile mappedFile) throws IllegalStateException {
        this(mappedFile, "");
    }

    protected MappedBytes(@NotNull MappedFile mappedFile, String name) throws IllegalStateException {
        super(NoBytesStore.noBytesStore(), NoBytesStore.noBytesStore().writePosition(),
                NoBytesStore.noBytesStore().writeLimit(), name);
        this.mappedFile = reserve(mappedFile);
        assert !mappedFile.isClosed();
        clear();
    }

    @NotNull
    private static MappedFile reserve(@NotNull MappedFile mappedFile) {
        mappedFile.reserve();
        return mappedFile;
    }

    @NotNull
    public static MappedBytes mappedBytes(@NotNull String filename, long chunkSize)
            throws FileNotFoundException, IllegalStateException {
        return mappedBytes(new File(filename), chunkSize);
    }

    @NotNull
    public static MappedBytes mappedBytes(@NotNull File file, long chunkSize)
            throws FileNotFoundException, IllegalStateException {
        return mappedBytes(file, chunkSize, OS.pageSize());
    }

    @NotNull
    public static MappedBytes mappedBytes(@NotNull File file, long chunkSize, long overlapSize)
            throws FileNotFoundException, IllegalStateException {
        @NotNull MappedFile rw = MappedFile.of(file, chunkSize, overlapSize, false);
        try {
            return mappedBytes(rw);
        } finally {
            rw.release();
        }
    }

    @NotNull
    public static MappedBytes mappedBytes(@NotNull File file,
                                          long chunkSize,
                                          long overlapSize,
                                          boolean readOnly) throws FileNotFoundException,
            IllegalStateException {
        @NotNull MappedFile rw = MappedFile.of(file, chunkSize, overlapSize, readOnly);
        try {
            return mappedBytes(rw);
        } finally {
            rw.release();
        }
    }

    @NotNull
    public static MappedBytes mappedBytes(@NotNull MappedFile rw) {
        return new MappedBytes(rw);
    }

    @NotNull
    public static MappedBytes readOnly(@NotNull File file) throws FileNotFoundException {
        return new MappedBytes(MappedFile.readOnly(file));
    }

    public void setNewChunkListener(NewChunkListener listener) {
        mappedFile.setNewChunkListener(listener);
    }

    @NotNull
    public MappedFile mappedFile() {
        return mappedFile;
    }

    @NotNull
    public MappedBytes withSizes(long chunkSize, long overlapSize) {
        @NotNull MappedFile mappedFile2 = this.mappedFile.withSizes(chunkSize, overlapSize);
        if (mappedFile2 == this.mappedFile)
            return this;
        try {
            return mappedBytes(mappedFile2);
        } finally {
            mappedFile2.release();
            release();
        }
    }

    @Override
    public BytesStore<Bytes<Void>, Void> copy() {
        return NativeBytes.copyOf(this);
    }

    @Override
    public long capacity() {
        return mappedFile == null ? 0L : mappedFile.capacity();
    }

    @Override
    public long realCapacity() {
        try {
            return mappedFile.actualSize();

        } catch (IORuntimeException e) {
            Jvm.warn().on(getClass(), "Unable to obtain the real size for " + mappedFile.file(), e);
            return 0;
        }
    }

    @Override
    protected void readCheckOffset(long offset, long adding, boolean given) throws BufferUnderflowException {
        long check = adding >= 0 ? offset : offset + adding;
        if (!bytesStore.inside(check)) {
            acquireNextByteStore(offset);
        }
        super.readCheckOffset(offset, adding, given);
    }

    @Override
    protected void writeCheckOffset(long offset, long adding) throws BufferOverflowException {
        if (offset < 0 || offset > capacity() - adding)
            throw writeIllegalArgumentException(offset);
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
//        super.writeCheckOffset(offset, adding);
    }

    @NotNull
    private IllegalArgumentException writeIllegalArgumentException(long offset) {
        return new IllegalArgumentException("Offset out of bound " + offset);
    }

    private void acquireNextByteStore(long offset) {
        if (bytesStore.inside(offset))
            return;

        try {
            @Nullable BytesStore newBS = mappedFile.acquireByteStore(offset);
            @Nullable BytesStore oldBS = bytesStore;
            bytesStore = newBS;
            oldBS.release();

        } catch (@NotNull IOException | IllegalStateException | IllegalArgumentException e) {
            @NotNull BufferOverflowException boe = new BufferOverflowException();
            boe.initCause(e);
            throw boe;
        }
    }

    @Nullable
    @Override
    public MappedBytesStore bytesStore() {
        return (MappedBytesStore) super.bytesStore();
    }

    @Override
    public long start() {
        return 0L;
    }

    @NotNull
    @Override
    public Bytes<Void> writePosition(long position) throws BufferOverflowException {
        if (position > writeLimit)
            throw new BufferOverflowException();
        if (position < 0L)
            throw new BufferUnderflowException();
        if (position < readPosition)
            this.readPosition = position;
        this.writePosition = position;
        return this;
    }

    @NotNull
    @Override
    public Bytes<Void> clear() {
        long start = 0L;
        readPosition = start;
        this.writePosition = start;
        writeLimit = mappedFile.capacity();
        return this;
    }

    @NotNull
    @Override
    public Bytes<Void> writeByte(byte i8) throws BufferOverflowException {
        long oldPosition = writePosition;
        if (writePosition < 0 || writePosition > capacity() - (long) 1)
            throw writeIllegalArgumentException(writePosition);
        if (!bytesStore.inside(writePosition)) {
            acquireNextByteStore(writePosition);
        }
        this.writePosition = writePosition + (long) 1;
        long offset = oldPosition;
        bytesStore.writeByte(offset, i8);
        return this;
    }

    @Override
    protected void performRelease() {
        super.performRelease();
        mappedFile.release();
    }

    @Override
    public boolean isElastic() {
        return true;
    }

    @NotNull
    @Override
    public Bytes<Void> write(@NotNull BytesStore bytes, long offset, long length)
            throws BufferUnderflowException, BufferOverflowException {
        if (length == 8) {
            writeLong(bytes.readLong(offset));

        } else if (bytes.isDirectMemory() && length >= 16) {
            rawCopy(bytes, offset, length);

        } else {
            BytesInternal.writeFully(bytes, offset, length, this);
        }
        return this;
    }

    public void rawCopy(@NotNull BytesStore bytes, long offset, long length)
            throws BufferOverflowException, BufferUnderflowException {
        long len = Math.min(writeRemaining(), Math.min(bytes.readRemaining(), length));
        if (len > 0) {
            OS.memory().copyMemory(bytes.address(offset), address(writePosition()), len);
            uncheckedWritePosition(writePosition() + len);
        }
    }

    @NotNull
    @Override
    public Bytes<Void> append8bit(@NotNull CharSequence cs, int start, int end)
            throws IllegalArgumentException, BufferOverflowException, BufferUnderflowException,
            IndexOutOfBoundsException {
        // check the start.
        long pos = writePosition();
        writeCheckOffset(pos, 0);
        if (!(cs instanceof String) || pos + (end - start) * 3 + 5 >= safeLimit()) {
            return super.append8bit(cs, start, end);
        }
        return append8bit0((String) cs, start, end - start);
    }

    @NotNull
    @Override
    public Bytes<Void> writeUtf8(String s) throws BufferOverflowException {
        char[] chars = extractChars(s);
        long utfLength = AppendableUtil.findUtf8Length(chars);
        writeStopBit(utfLength);
        appendUtf8(chars, 0, chars.length);
        return this;
    }

    @NotNull
    public MappedBytes write8bit(CharSequence s, int start, int length) {
        // check the start.
        long pos = writePosition();
        writeCheckOffset(pos, 0);
        if (!(s instanceof String) || pos + length * 3 + 5 >= safeLimit()) {
            super.write8bit(s, start, length);
            return this;
        }

        writeStopBit(length);
        return append8bit0((String) s, start, length);
    }

    @NotNull
    private MappedBytes append8bit0(String s, int start, int length) {
        char[] chars = StringUtils.extractChars(s);
        long address = address(writePosition());
        Memory memory = bytesStore().memory;
        int i = 0;
        for (; i < length - 3; i += 4) {
            int c0 = chars[i + start] & 0xff;
            int c1 = chars[i + start + 1] & 0xff;
            int c2 = chars[i + start + 2] & 0xff;
            int c3 = chars[i + start + 3] & 0xff;
            memory.writeInt(address, (c3 << 24) | (c2 << 16) | (c1 << 8) | c0);
            address += 4;
        }
        for (; i < length; i++) {
            char c = chars[i + start];
            memory.writeByte(address++, (byte) c);
        }
        writeSkip(length);
        return this;
    }

    @NotNull
    @Override
    public Bytes<Void> appendUtf8(CharSequence cs, int start, int length)
            throws BufferOverflowException, IllegalArgumentException {
        // check the start.
        long pos = writePosition();
        writeCheckOffset(pos, 0);
        if (!(cs instanceof String) || pos + length * 3 + 5 >= safeLimit()) {
            super.appendUtf8(cs, start, length);
            return this;
        }

        char[] chars = StringUtils.extractChars((String) cs);
        long address = address(pos);
        Memory memory = OS.memory();
        int i = 0;
        non_ascii:
        {
            for (; i < length; i++) {
                char c = chars[i + start];
                if (c > 127) {
                    writeSkip(i);
                    break non_ascii;
                }
                memory.writeByte(address++, (byte) c);
            }
            writeSkip(length);
            return this;
        }
        for (; i < length; i++) {
            char c = chars[i + start];
            appendUtf8(c);
        }
        return this;
    }

    @Override
    public boolean sharedMemory() {
        return true;
    }

    @NotNull
    public Bytes<Void> writeOrderedInt(long offset, int i) throws BufferOverflowException {
        assert writeCheckOffset0(offset, (long) 4);
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
        bytesStore.writeOrderedInt(offset, i);
        return this;
    }

    @Override
    public byte readVolatileByte(long offset) throws BufferUnderflowException {
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
        return bytesStore.readVolatileByte(offset);
    }

    @Override
    public short readVolatileShort(long offset) throws BufferUnderflowException {
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
        return bytesStore.readVolatileShort(offset);
    }

    @Override
    public int readVolatileInt(long offset) throws BufferUnderflowException {
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
        return bytesStore.readVolatileInt(offset);
    }

    @Override
    public long readVolatileLong(long offset) throws BufferUnderflowException {
        if (!bytesStore.inside(offset)) {
            acquireNextByteStore(offset);
        }
        return bytesStore.readVolatileLong(offset);
    }

    @Override
    public int peekVolatileInt() {
        if (!bytesStore.inside(readPosition)) {
            acquireNextByteStore(readPosition);
        }

        @Nullable MappedBytesStore bytesStore = (MappedBytesStore) (BytesStore) this.bytesStore;
        long address = bytesStore.address + bytesStore.translate(readPosition);
        @Nullable Memory memory = bytesStore.memory;

        // are we inside a cache line?
        if ((address & 63) <= 60) {
            if (memory == null)
                throw new NullPointerException();

            int value = UnsafeMemory.UNSAFE.getIntVolatile(null, address);
            if (value != 0 && value != 0x80000000)
                return value;
            return UnsafeMemory.UNSAFE.getIntVolatile(null, address);

        } else {
            return memory.readVolatileInt(address);
        }
    }

    @Override
    public void close() {
        this.release();
    }

    @Override
    public boolean isClosed() {
        return this.refCount() <= 0;
    }

    @NotNull
    @Override
    public MappedBytes write(@NotNull BytesStore bytes)
            throws BufferOverflowException {
        assert bytes != this : "you should not write to yourself !";

        long offset = bytes.readPosition();
        long length = Math.min(writeRemaining(), bytes.readRemaining());
        if (length == 8) {
            writeLong(bytes.readLong(offset));

        } else if (bytes.isDirectMemory() && length >= 16 && length <= writeRemaining()) {
            rawCopy(bytes, offset, length);

        } else {
            BytesInternal.writeFully(bytes, offset, length, this);
        }
        return this;
    }
}
