/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ElementSupport;
import net.shibboleth.spring.http.InMemoryCachingHttpClientFactoryBean;

/**
 * Parser for abstract dynamic HTTP metadata resolvers.
 */
public abstract class AbstractDynamicHTTPMetadataProviderParser extends AbstractDynamicMetadataProviderParser {

    /** Default caching type . */
    @Nonnull private static final Class<?> DEFAULT_CACHING_CLASS = InMemoryCachingHttpClientFactoryBean.class;

    /** Default max total connections. */
    @Nonnull private static final Integer DEFAULT_MAX_CONNECTIONS_TOTAL = 100;

    /** Default max connections per route. */
    @Nonnull private static final Integer DEFAULT_MAX_CONNECTIONS_PER_ROUTE = 100;

    /** Default request connection timeout. */
    @Nonnull @NotEmpty private static final String DEFAULT_CONNECTION_REQUEST_TIMEOUT = "PT5S";

    /** Default connection timeout. */
    @Nonnull @NotEmpty private static final String DEFAULT_CONNECTION_TIMEOUT = "PT5S";

    /** Default socket timeout. */
    @Nonnull @NotEmpty private static final String DEFAULT_SOCKET_TIMEOUT = "PT5S";

    /** Logger. */
    private final Logger log = LoggerFactory.getLogger(AbstractDynamicHTTPMetadataProviderParser.class);

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override protected void doNativeParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doNativeParse(element, parserContext, builder);

        final BeanDefinition httpClientSecurityParameters =
                HTTPMetadataProvidersParserSupport.buildHttpClientSecurityParameters(
                        ElementSupport.getFirstChildElement(element,
                                HTTPMetadataProvidersParserSupport.TLS_TRUST_ENGINE_ELEMENT_NAME),
                        StringSupport.trimOrNull(element.getAttributeNS(null, "httpClientSecurityParametersRef")),
                        parserContext);

        builder.addPropertyValue("httpClientSecurityParameters", httpClientSecurityParameters);

        if (element.hasAttributeNS(null, "httpClientRef")) {
            builder.addConstructorArgReference(
                    AttributeSupport.ensureAttributeValue(element, null, "httpClientRef"));
            if (element.hasAttributeNS(null, "connectionTimeout")
                    || element.hasAttributeNS(null, "connectionRequestTimeout")
                    || element.hasAttributeNS(null, "socketTimeout")
                    || element.hasAttributeNS(null, "maxConnectionsTotal")
                    || element.hasAttributeNS(null, "maxConnectionsPerRoute")
                    || element.hasAttributeNS(null, "disregardTLSCertificate")
                    || element.hasAttributeNS(null, "proxyHost") || element.hasAttributeNS(null, "proxyPort")
                    || element.hasAttributeNS(null, "proxyUser") || element.hasAttributeNS(null, "proxyPassword")) {
                log.warn("httpClientRef overrides settings for connectionTimeout, "
                    + "connectionRequestTimeout, socketTimeout, maxConnectionsTotal, maxConnectionsPerRoute, "
                    + "disregardTLSCertificate, proxyHost, proxyPort, "
                    + "proxyUser and proxyPassword");
            }
        } else {
            builder.addConstructorArgValue(buildHttpClient(element, parserContext, httpClientSecurityParameters));
        }

        if (element.hasAttributeNS(null, "supportedContentTypes")) {
            final AbstractBeanDefinition listBuilder =
                    SpringSupport.getAttributeValueAsList(element.getAttributeNodeNS(null, "supportedContentTypes"));
            builder.addPropertyValue("supportedContentTypes", listBuilder);
        }
    }
// Checkstyle: CyclomaticComplexity ON

    /**
     * Build the definition of the HTTPClientBuilder which contains all our configuration.
     * 
     * @param element the HTTPMetadataProvider parser.
     * @param parserContext the context
     * @param httpClientSecurityParameters the client security parameters to be used
     * @return the bean definition with the parameters.
     * 
     * Either httpClientSecurityParametersRef or httpClientSecurityParameters can be present, not both.
     */
    @Nonnull private BeanDefinition buildHttpClient(@Nonnull final Element element,
            @Nonnull final ParserContext parserContext, @Nullable final BeanDefinition httpClientSecurityParameters) {

        final BeanDefinitionBuilder clientBuilder = 
                HTTPMetadataProvidersParserSupport.buildCommonClientBuilder(element,
                        parserContext,
                        DEFAULT_CACHING_CLASS ,
                        httpClientSecurityParameters);

        // Set up non standard defaults
        if (!element.hasAttributeNS(null, "connectionTimeout")) {
            clientBuilder.addPropertyValue("connectionTimeout", DEFAULT_CONNECTION_TIMEOUT);
        }

        if (!element.hasAttributeNS(null, "connectionRequestTimeout")) {
            clientBuilder.addPropertyValue("connectionRequestTimeout", DEFAULT_CONNECTION_REQUEST_TIMEOUT);
        }
        if (!element.hasAttributeNS(null, "socketTimeout")) {
            clientBuilder.addPropertyValue("socketTimeout", DEFAULT_SOCKET_TIMEOUT);
        }

        if (element.hasAttributeNS(null, "maxConnectionsTotal")) {
            clientBuilder.addPropertyValue("maxConnectionsTotal",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "maxConnectionsTotal")));
        } else {
            clientBuilder.addPropertyValue("maxConnectionsTotal", DEFAULT_MAX_CONNECTIONS_TOTAL);
        }

        // set up non-common attributes
        if (element.hasAttributeNS(null, "maxConnectionsPerRoute")) {
            clientBuilder.addPropertyValue("maxConnectionsPerRoute",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "maxConnectionsPerRoute")));
        } else {
            clientBuilder.addPropertyValue("maxConnectionsPerRoute", DEFAULT_MAX_CONNECTIONS_PER_ROUTE);
        }

        return clientBuilder.getBeanDefinition();
    }

}