/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.persist.FilesystemLoadSaveManager;
import org.slf4j.Logger;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ParserPool;

/**
 * Parser for {@link org.opensaml.saml.metadata.resolver.impl.AbstractDynamicMetadataResolver}.
 */
public abstract class AbstractDynamicMetadataProviderParser extends AbstractMetadataProviderParser {
    
    /** Logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(AbstractDynamicMetadataProviderParser.class);

    /** Bean ID for defaulted {@link ParserPool} instance. */
    @Nullable private final String parserPoolRef;
    
    /** Constructor. */
    public AbstractDynamicMetadataProviderParser() {
        // Will warn later if not set.
        parserPoolRef =
                getCustomProperty(AbstractDynamicMetadataProviderParser.class.getName() + ".ParserPool.bean", null);
    }
    
    /**
     * 
     * {@inheritDoc}
     * 
     * We assume that we will be summoning up a class which extends an
     * {@link org.opensaml.saml.metadata.resolver.impl.AbstractDynamicMetadataResolver}.
     */
    @Override protected void doNativeParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext, 
            @Nonnull final BeanDefinitionBuilder builder) {

        super.doNativeParse(element, parserContext, builder);

        // If there's a timer bean reference, that's the first c'tor argument.
        final String timerRef = getTaskTimerRef(element);
        if (timerRef != null) {
            builder.addConstructorArgReference(timerRef);
        }

        processTimingProperties(element, parserContext, builder);

        if (element.hasAttributeNS(null, "expirationWarningThreshold")) {
            builder.addPropertyValue("expirationWarningThreshold",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "expirationWarningThreshold")));
        }

        if (element.hasAttributeNS(null, "indexesRef")) {
            builder.addPropertyReference("indexes",
                    AttributeSupport.ensureAttributeValue(element, null, "indexesRef"));
        }
        
        builder.addPropertyReference("parserPool", getParserPoolRef(element));

        processPersistentCachingProperties(element, parserContext, builder);
    }

    /**
     * Process options related to time.
     *
     * @param element current element
     * @param parserContext current parser context
     * @param builder current builder
     */
    protected void processTimingProperties(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {

        if (element.hasAttributeNS(null, "refreshDelayFactor")) {
            builder.addPropertyValue("refreshDelayFactor",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "refreshDelayFactor")));
        }

        if (element.hasAttributeNS(null, "minCacheDuration")) {
            builder.addPropertyValue("minCacheDuration",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "minCacheDuration")));
        }

        if (element.hasAttributeNS(null, "maxCacheDuration")) {
            builder.addPropertyValue("maxCacheDuration",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "maxCacheDuration")));
        }

        if (element.hasAttributeNS(null, "negativeLookupCacheDuration")) {
            builder.addPropertyValue("negativeLookupCacheDuration",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "negativeLookupCacheDuration")));
        }

        if (element.hasAttributeNS(null, "maxIdleEntityData")) {
            builder.addPropertyValue("maxIdleEntityData",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "maxIdleEntityData")));
        }

        if (element.hasAttributeNS(null, "removeIdleEntityData")) {
            builder.addPropertyValue("removeIdleEntityData",
                    SpringSupport.getStringValueAsBoolean(element.getAttributeNS(null, "removeIdleEntityData")));
        }

        if (element.hasAttributeNS(null, "cleanupTaskInterval")) {
            builder.addPropertyValue("cleanupTaskInterval",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "cleanupTaskInterval")));
        }
    }
    
    /**
     * Process options related to persistent caching support.
     *
     * @param element current element
     * @param parserContext current parser context
     * @param builder current builder
     */
    protected void processPersistentCachingProperties(final Element element, final ParserContext parserContext, 
            final BeanDefinitionBuilder builder) {
        
        if (element.hasAttributeNS(null, "persistentCacheManagerRef")) {
            builder.addPropertyReference("persistentCacheManager",
                    AttributeSupport.ensureAttributeValue(element, null, "persistentCacheManagerRef"));
            
            if (element.hasAttributeNS(null, "persistentCacheManagerDirectory")) {
                log.warn("{} Element contained both persistentCacheManagerRef and persistentCacheManagerDirectory, " 
                        + "persistentCacheManagerDirectory will be ignored", 
                        parserContext.getReaderContext().getResource().getDescription());
            }
            
        } else if (element.hasAttributeNS(null, "persistentCacheManagerDirectory")) {
            final String cacheDirectory = 
                    StringSupport.trimOrNull(element.getAttributeNS(null, "persistentCacheManagerDirectory"));
            log.debug("{} Building internally-constructed FilesystemLoadSaveManager with base directory: {}", 
                    parserContext.getReaderContext().getResource().getDescription(), cacheDirectory);
            final BeanDefinitionBuilder cacheManagerBuilder =
                    BeanDefinitionBuilder.genericBeanDefinition(FilesystemLoadSaveManager.class);
            cacheManagerBuilder.addConstructorArgValue(cacheDirectory);
            builder.addPropertyValue("persistentCacheManager", cacheManagerBuilder.getBeanDefinition());
        }
        
        if (element.hasAttributeNS(null, "persistentCacheKeyGeneratorRef")) {
            builder.addPropertyReference("persistentCacheKeyGenerator",
                    AttributeSupport.ensureAttributeValue(element, null, "persistentCacheKeyGeneratorRef"));
        }
        
        if (element.hasAttributeNS(null, "initializeFromPersistentCacheInBackground")) {
            builder.addPropertyValue("initializeFromPersistentCacheInBackground", SpringSupport.getStringValueAsBoolean(
                            element.getAttributeNS(null, "initializeFromPersistentCacheInBackground")));
        }
        
        if (element.hasAttributeNS(null, "backgroundInitializationFromCacheDelay")) {
            builder.addPropertyValue("backgroundInitializationFromCacheDelay",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "backgroundInitializationFromCacheDelay")));
        }
        
        if (element.hasAttributeNS(null, "initializationFromCachePredicateRef")) {
            builder.addPropertyReference("initializationFromCachePredicate",
                    AttributeSupport.ensureAttributeValue(element, null, "initializationFromCachePredicateRef"));
        }
    }

    /**
     * Gets the default task timer reference for the metadata provider.
     * 
     * @param element metadata provider configuration element
     * 
     * @return task timer reference
     */
    @Nullable protected String getTaskTimerRef(@Nonnull final Element element) {

        if (element.hasAttributeNS(null, "taskTimerRef")) {
            return AttributeSupport.ensureAttributeValue(element, null, "taskTimerRef");
        }
        return null;
    }

    /**
     * Gets the default parser pool reference for the metadata provider.
     * 
     * @param element metadata provider configuration element
     * 
     * @return parser pool reference
     */
    @Nonnull @NotEmpty protected String getParserPoolRef(@Nonnull final Element element) {
        if (element.hasAttributeNS(null, "parserPoolRef")) {
            return AttributeSupport.ensureAttributeValue(element, null, "parserPoolRef");
        }

        if (parserPoolRef != null) {
            return parserPoolRef;
        }
        
        throw new BeanCreationException("Default ParserPool bean ID not available.");
    }

}