/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ParserPool;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

/**
 * Parser for all types which extend the &lt;ReloadingMetadataProviderType&gt;.
 */
public abstract class AbstractReloadingMetadataProviderParser extends AbstractMetadataProviderParser {

    /** The reference to the system parser pool that we set up. */
    @Nonnull @NotEmpty private static final String DEFAULT_PARSER_POOL_REF = "shibboleth.ParserPool";

    /** Bean ID for defaulted {@link ParserPool} instance. */
    @Nullable private final String parserPoolRef;
    
    /** Constructor. */
    public AbstractReloadingMetadataProviderParser() {
        // Will warn later if not set.
        parserPoolRef =
                getCustomProperty(AbstractReloadingMetadataProviderParser.class.getName() + ".ParserPool.bean", null);
    }
    
    /**
     * 
     * {@inheritDoc}
     * 
     * We assume in setting the that we will be summoning up a class which extends 
     * {@link org.opensaml.saml.metadata.resolver.impl.AbstractReloadingMetadataResolver}.
     */
    @Override protected void doNativeParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {

        super.doNativeParse(element, parserContext, builder);

        // If there's a timer bean reference, that's the first c'tor argument.
        final String timerRef = getTaskTimerRef(element);
        if (timerRef != null) {
            builder.addConstructorArgReference(timerRef);
        }

        builder.addPropertyReference("parserPool", getParserPoolRef(element));
        
        if (element.hasAttributeNS(null, "indexesRef")) {
            builder.addPropertyReference("indexes",
                    AttributeSupport.ensureAttributeValue(element, null, "indexesRef"));
        }

        if (element.hasAttributeNS(null, "resolveViaPredicatesOnly")) {
            builder.addPropertyValue("resolveViaPredicatesOnly", SpringSupport.getStringValueAsBoolean(
                    element.getAttributeNS(null, "resolveViaPredicatesOnly")));
        }

        if (element.hasAttributeNS(null, "refreshDelayFactor")) {
            builder.addPropertyValue("refreshDelayFactor",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "refreshDelayFactor")));
        }

        if (element.hasAttributeNS(null, "maxRefreshDelay")) {
            builder.addPropertyValue("maxRefreshDelay",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "maxRefreshDelay")));
        }

        if (element.hasAttributeNS(null, "minRefreshDelay")) {
            builder.addPropertyValue("minRefreshDelay",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "minRefreshDelay")));
        }

        if (element.hasAttributeNS(null, "expirationWarningThreshold")) {
            builder.addPropertyValue("expirationWarningThreshold",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "expirationWarningThreshold")));
        }
        
    }

    /**
     * Gets the default task timer reference for the metadata provider.
     * 
     * @param element metadata provider configuration element
     * 
     * @return task timer reference
     */
    @Nullable protected String getTaskTimerRef(final Element element) {
        if (element.hasAttributeNS(null, "taskTimerRef")) {
            return AttributeSupport.ensureAttributeValue(element, null, "taskTimerRef");
        }
        return null;
    }

    /**
     * Gets the default parser pool reference for the metadata provider.
     * 
     * @param element metadata provider configuration element
     * 
     * @return parser pool reference
     */
    @Nonnull @NotEmpty protected String getParserPoolRef(final Element element) {
        if (element.hasAttributeNS(null, "parserPoolRef")) {
            return AttributeSupport.ensureAttributeValue(element, null, "parserPoolRef");
        }

        if (parserPoolRef != null) {
            return parserPoolRef;
        }
        
        throw new BeanCreationException("Default ParserPool bean ID not available.");
    }

}