/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ElementSupport;
import net.shibboleth.spring.http.HttpClientFactoryBean;

/**
 * Parser for a FilesystemMetadataProvider.
 */
public abstract class HTTPMetadataProviderParser extends AbstractReloadingMetadataProviderParser {

    /** Default caching type. */
    @Nonnull private static final Class<?> DEFAULT_CLIENT_CLASS = HttpClientFactoryBean.class;

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPMetadataProviderParser.class);

    /** {@inheritDoc} */
    @Override protected void doNativeParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        
        super.doNativeParse(element, parserContext, builder);

        final BeanDefinition httpClientSecurityParameters =
                HTTPMetadataProvidersParserSupport.buildHttpClientSecurityParameters(
                        ElementSupport.getFirstChildElement(element,
                                HTTPMetadataProvidersParserSupport.TLS_TRUST_ENGINE_ELEMENT_NAME),
                        StringSupport.trimOrNull(element.getAttributeNS(null, "httpClientSecurityParametersRef")),
                        parserContext);

        builder.addPropertyValue("httpClientSecurityParameters", httpClientSecurityParameters);

        if (element.hasAttributeNS(null, "httpClientRef")) {
            builder.addConstructorArgReference(
                    AttributeSupport.ensureAttributeValue(element, null, "httpClientRef"));
            if (element.hasAttributeNS(null, "connectionTimeout")
                    || element.hasAttributeNS(null, "connectionRequestTimeout")
                    || element.hasAttributeNS(null, "socketTimeout")
                    || element.hasAttributeNS(null, "disregardTLSCertificate")
                    || element.hasAttributeNS(null, "proxyHost") || element.hasAttributeNS(null, "proxyPort")
                    || element.hasAttributeNS(null, "proxyUser") || element.hasAttributeNS(null, "proxyPassword")) {
                log.warn("httpClientRef overrides settings for connectionTimeout, " 
                    + "connectionRequestTimeout, socketTimeout, disregardTLSCertificate, "
                    + " proxyHost, proxyPort, proxyUser and proxyPassword");
            }
        } else {
            builder.addConstructorArgValue(buildHttpClient(element, parserContext, httpClientSecurityParameters));
        }
        builder.addConstructorArgValue(StringSupport.trimOrNull(element.getAttributeNS(null,
                HTTPMetadataProvidersParserSupport.METADATA_URL)));
    }

    /**
     * Build the definition of the HTTPClientBuilder which contains all our configuration.
     * 
     * @param element the HTTPMetadataProvider parser.
     * @param parserContext context
     * @param httpClientSecurityParameters the client security parameters to be used
     * @return the bean definition with the parameters.
     */
    @Nonnull private BeanDefinition buildHttpClient(@Nonnull final Element element,
            @Nonnull final ParserContext parserContext, @Nullable final BeanDefinition httpClientSecurityParameters) {

        return HTTPMetadataProvidersParserSupport.buildCommonClientBuilder(element, parserContext,
                HTTPMetadataProviderParser.DEFAULT_CLIENT_CLASS, httpClientSecurityParameters).getBeanDefinition();
    }
    
}