/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.opensaml.security.httpclient.HttpClientSecurityParameters;
import org.opensaml.spring.httpclient.HttpClientSecurityParametersMergingFactoryBean;
import org.opensaml.spring.tls.TLSSocketFactoryFactoryBean;
import org.slf4j.Logger;
import org.springframework.beans.BeanMetadataElement;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.ManagedList;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.ElementSupport;
import net.shibboleth.spring.http.HttpClientFactoryBean;

/**
 * Helper class for Spring configuration of HTTP metadata providers.
 */
public final class HTTPMetadataProvidersParserSupport {
    
    /** The URL for the metadata. */
    @Nonnull @NotEmpty public static final String METADATA_URL = "metadataURL";

    /** TLSTrustEngine element name. */
    @Nonnull public static final QName TLS_TRUST_ENGINE_ELEMENT_NAME =
            new QName(AbstractMetadataProviderParser.METADATA_NAMESPACE, "TLSTrustEngine");

    /** Class logger. */
    @Nonnull private static final Logger LOG = LoggerFactory.getLogger(HTTPMetadataProvidersParserSupport.class);

    /** Constructor. */
    private HTTPMetadataProvidersParserSupport() {
    }
    
    /**
     * Build the definition of the HttpClientSecurityParameters as a factory bean which merges the inline
     * TLS TrustEngine and parameters ref inputs.
     * 
     * @param tlsTrustEngine the element defining the inline TLS TrustEngine
     * @param parametersRef the security parameters bean reference
     * @param parserContext context
     * @return the bean definition with the parameters.
     */
    @Nonnull protected static BeanDefinition buildHttpClientSecurityParameters(@Nullable final Element tlsTrustEngine,
            @Nullable final String parametersRef, @Nonnull final ParserContext parserContext) {
        
        final BeanDefinitionBuilder factoryBuilder =
                BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParametersMergingFactoryBean.class);
        
        final List<BeanMetadataElement> factoryInputs = new ManagedList<>(2);

        // First order-of-precedence
        if (tlsTrustEngine != null)  {
            final BeanDefinition trustEngineParams = parseTLSTrustEngine(tlsTrustEngine, parserContext);
            if (trustEngineParams != null) {
                factoryInputs.add(trustEngineParams);
            }
        }

        // Second order-of-precedence
        if (parametersRef != null) {
            factoryInputs.add(new RuntimeBeanReference(parametersRef));
        }

        factoryBuilder.addPropertyValue("parameters", factoryInputs);
        
        return factoryBuilder.getBeanDefinition();
    }
    
    /**
     * Build the definition of the HTTPClientBuilder which contains all our configuration.
     * 
     * @param element the HTTPMetadataProvider parser.
     * @param parserContext context
     * @param httpClientSecurityParameters a bean definition
     * @return the bean definition with the parameters.
     */
    @Nonnull protected static BeanDefinition buildTLSSocketFactory(
            @Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nullable final BeanDefinition httpClientSecurityParameters) {

        Constraint.isNotNull(LOG, "LOG must be present");
        final BeanDefinitionBuilder tlsSocketFactoryBuilder = 
                BeanDefinitionBuilder.genericBeanDefinition(TLSSocketFactoryFactoryBean.class);

        if (element.hasAttributeNS(null, "disregardTLSCertificate")) {
            tlsSocketFactoryBuilder.addPropertyValue("connectionDisregardTLSCertificate",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "disregardTLSCertificate")));
        }

        if (httpClientSecurityParameters != null) {
            tlsSocketFactoryBuilder.addPropertyValue("httpClientSecurityParameters", 
                    httpClientSecurityParameters);
        }

        return tlsSocketFactoryBuilder.getBeanDefinition();
    }

    /**
     * Build the BeanDefinition of the {@link HttpClientSecurityParameters} which contains the TLS trust engine
     * provided.
     * 
     * <p>One of the first two parameters must be non-null.</p>
     * 
     * @param tlsTrustEngine if present, the TLSTrustEngine element
     * @param parserContext context
     * 
     * @return the bean definition
     */
    @Nullable protected static BeanDefinition parseTLSTrustEngine(@Nullable final Element tlsTrustEngine,
            @Nonnull final ParserContext parserContext) {

        final BeanDefinitionBuilder builder =
                BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParameters.class);
        if (tlsTrustEngine != null) {
            final Element trustEngine = ElementSupport.getFirstChildElement(tlsTrustEngine,
                            AbstractMetadataProviderParser.TRUST_ENGINE_ELEMENT_NAME);

            if (trustEngine == null) {
                // This should be schema-invalid, but LOG a warning just in case.
                LOG.warn("{}:, Element {} did not contain a {} child element", 
                        parserContext.getReaderContext().getResource().getDescription(), 
                        TLS_TRUST_ENGINE_ELEMENT_NAME,
                        AbstractMetadataProviderParser.TRUST_ENGINE_ELEMENT_NAME);
                return null;
            }
            builder.addPropertyValue("tLSTrustEngine", 
                    SpringSupport.parseCustomElement(trustEngine, parserContext, builder, false));
        }

        return builder.getBeanDefinition();
    }

    /**
     * Build a {@link BeanDefinitionBuilder} for a {@link HttpClientFactoryBean} and populate it from the "standard"
     * attributes which are shared between the Dynamic and Static providers. non standard defaults are applied by the
     * caller.
     * 
     * @param element the configuration
     * @param parserContext context
     * @param clientBuildClass the type of builder to create.
     * @param httpClientSecurityParameters the client security parameters to be used
     * 
     * @return an appropriate builder
     */
    @Nonnull protected static BeanDefinitionBuilder buildCommonClientBuilder(@Nonnull final Element element,
            @Nonnull final ParserContext parserContext, @Nonnull final Class<?> clientBuildClass,
            @Nullable final BeanDefinition httpClientSecurityParameters) {

        final BeanDefinitionBuilder clientBuilder = BeanDefinitionBuilder.genericBeanDefinition(clientBuildClass);
        clientBuilder.setLazyInit(true);

        if (element.hasAttributeNS(null, "connectionTimeout")) {
            clientBuilder.addPropertyValue("connectionTimeout",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "connectionTimeout")));
        }
        if (element.hasAttributeNS(null, "connectionRequestTimeout")) {
            clientBuilder.addPropertyValue("connectionRequestTimeout",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "connectionRequestTimeout")));
        }
        if (element.hasAttributeNS(null, "socketTimeout")) {
            clientBuilder.addPropertyValue("socketTimeout",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "socketTimeout")));
        }

        clientBuilder.addPropertyValue("tLSSocketFactory",
                HTTPMetadataProvidersParserSupport.buildTLSSocketFactory(
                        element, parserContext, httpClientSecurityParameters));

        if (element.hasAttributeNS(null, "proxyHost")) {
            clientBuilder.addPropertyValue("connectionProxyHost",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "proxyHost")));
        }

        if (element.hasAttributeNS(null, "proxyPort")) {
            clientBuilder.addPropertyValue("connectionProxyPort",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "proxyPort")));
        }

        if (element.hasAttributeNS(null, "proxyUser")) {
            clientBuilder.addPropertyValue("connectionProxyUsername",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "proxyUser")));
        }

        if (element.hasAttributeNS(null, "proxyPassword")) {
            clientBuilder.addPropertyValue("connectionProxyPassword", element.getAttributeNS(null, "proxyPassword"));
        }

        return clientBuilder;
    }
    
}