/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.saml.metadata.resolver.ChainingMetadataResolver;
import org.opensaml.saml.metadata.resolver.MetadataResolver;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilter;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilterChain;
import org.opensaml.saml.metadata.resolver.filter.MetadataNodeProcessor;
import org.opensaml.saml.metadata.resolver.filter.impl.NodeProcessingMetadataFilter;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.Ordered;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.ComponentInitializationException;

/**
 * A {@link BeanPostProcessor} for {@link MetadataResolver} beans that ensures a {@link NodeProcessingMetadataFilter}
 * containing a set of default {@link MetadataNodeProcessor} plugins is attached.
 * 
 * <p>
 * This is done to ensure that other components function correctly, such as the PKIX trust engine and predicates that
 * depend on group information.
 * </p>
 * 
 * <p>The constructor will auto-wire all free-standing beans, but the property setter can override these.</p>
 */
public class NodeProcessingAttachingBeanPostProcessor implements BeanPostProcessor, Ordered {

    /** The processors to install. */
    @Nonnull private List<MetadataNodeProcessor> nodeProcessors;

    /**
     * Constructor.
     *
     * @param processors auto-wired processors to install
     */
    @Autowired
    public NodeProcessingAttachingBeanPostProcessor(@Nullable final Collection<MetadataNodeProcessor> processors) {
        if (processors != null) {
            nodeProcessors = CollectionSupport.copyToList(processors);
        } else {
            nodeProcessors = CollectionSupport.emptyList();
        }
    }
    
    /**
     * Set the {@link MetadataNodeProcessor} instances to auto-attach instead of the auto-wired set.
     * 
     * @param processors processors to auto-attach
     * 
     * @since 4.1.0
     */
    public void setNodeProcessors(@Nullable final Collection<MetadataNodeProcessor> processors) {
        if (processors != null) {
            // Replace auto-wired set.
            nodeProcessors = CollectionSupport.copyToList(processors);
        }
    }

    /** {@inheritDoc} */
    public int getOrder() {
        return LOWEST_PRECEDENCE;
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull public Object postProcessBeforeInitialization(@Nonnull final Object bean,
            @Nonnull final String beanName) {
        
        if (nodeProcessors.isEmpty()) {
            return bean;
        }
        
        // Do not attach to beans which just include other ones.
        if (!(bean instanceof MetadataResolver) || bean instanceof ChainingMetadataResolver) {
            return bean;
        }

        final MetadataResolver resolver = (MetadataResolver) bean;

        final NodeProcessingMetadataFilter filterToAttach = new NodeProcessingMetadataFilter();
        filterToAttach.setNodeProcessors(nodeProcessors);
        try {
            filterToAttach.initialize();
        } catch (final ComponentInitializationException e) {
            throw new BeanCreationException("Error initializing NodeProcessingMetadataFilter", e);
        }

        final MetadataFilter filter = resolver.getMetadataFilter();
        if (filter == null) {
            resolver.setMetadataFilter(filterToAttach);
        } else if (filter instanceof MetadataFilterChain) {
            ((MetadataFilterChain) filter).getFilters().add(filterToAttach);
        } else {
            final MetadataFilterChain chain = new MetadataFilterChain();
            chain.setFilters(CollectionSupport.listOf(filter, filterToAttach));
            resolver.setMetadataFilter(chain);
        }

        return resolver;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Object postProcessAfterInitialization(@Nonnull final Object bean, @Nonnull final String beanName) {
        return bean;
    }

}