/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.metadata.filter;

import java.util.List;

import javax.annotation.Nonnull;
import javax.xml.namespace.QName;

import org.opensaml.saml.metadata.resolver.filter.MetadataFilterChain;
import org.opensaml.saml.metadata.resolver.filter.impl.ByReferenceMetadataFilter;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.ManagedList;
import org.springframework.beans.factory.support.ManagedMap;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.ElementSupport;
import net.shibboleth.spring.metadata.AbstractMetadataProviderParser;

/**
 * Parser for a &lt;ByReference&gt; filter.
 */
public class ByReferenceParser extends AbstractMetadataFilterParser {

    /** Element name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AbstractMetadataProviderParser.METADATA_NAMESPACE, "ByReference");

    /** {@inheritDoc} */
    @Override
    @Nonnull protected Class<?> getBeanClass(@Nonnull final Element element) {
        return ByReferenceMetadataFilter.class;
    }

    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(element, parserContext, builder);

        final List<Element> children = ElementSupport.getChildElements(element,
                new QName(AbstractMetadataProviderParser.METADATA_NAMESPACE, "MetadataFilters"));

        if (null != children && !children.isEmpty()) {
            
            final ManagedMap<Object,BeanDefinition> mappings = new ManagedMap<>();
            
            for (final Element child : children) {
                assert child != null;
                final List<Element> filters = ElementSupport.getChildElements(child,
                        new QName(AbstractMetadataProviderParser.METADATA_NAMESPACE, "MetadataFilter"));
                if (filters != null && !filters.isEmpty()) {
                    final AbstractBeanDefinition providerRef =
                            SpringSupport.getAttributeValueAsList(child.getAttributeNodeNS(null, "providerRef"));
                    
                    final ManagedList<BeanDefinition> filterBeans =
                            SpringSupport.parseCustomElements(filters, parserContext, builder);
                    if (filterBeans != null) {
                        if (filterBeans.size() == 1) {
                            mappings.put(providerRef, filterBeans.get(0));
                        } else {
                            final BeanDefinitionBuilder chain =
                                    BeanDefinitionBuilder.genericBeanDefinition(MetadataFilterChain.class);
                            chain.addPropertyValue("filters", filterBeans);
                            mappings.put(providerRef, chain.getBeanDefinition());
                        }
                    }
                }
            }
            
            builder.addPropertyValue("filterMappings", mappings);
        }
    }

    /** {@inheritDoc} */
    @Override protected boolean shouldGenerateId() {
        return true;
    }

}