/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.security;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.security.credential.Credential;

import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;

/**
 * This is a utility class used as an auto-wiring source for collections of
 * signing and encryption {@link Credential} objects so that other layers of the
 * system can gain access to the complete set of them.
 * 
 * @since 5.0.0
 */
public class CredentialHolder {

    /** Credentials to expose. */
    @Nonnull private final Collection<Credential> credentials;
    
    /**
     * Constructor.
     *
     * @param creds credentials to expose to other components
     */
    public CredentialHolder(@Nullable final Collection<Credential> creds) {
        if (creds != null) {
            credentials = CollectionSupport.copyToList(creds);
        } else {
            credentials = CollectionSupport.emptyList();
        }
    }
    
    /**
     * Get the credentials to expose to other components.
     * 
     * @return credentials to expose
     */
    @Nonnull @Unmodifiable @NotLive public Collection<Credential> getCredentials() {
        return credentials;
    }
    
}