/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.security.trust;

import javax.annotation.Nonnull;
import javax.xml.namespace.QName;

import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.custom.AbstractCustomBeanDefinitionParser;
import net.shibboleth.spring.security.SecurityNamespaceHandler;

import org.opensaml.security.x509.PKIXValidationOptions;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

/**
 * A Parser for the &lt; ValidationOptions &gt; within a StaticPKIXSignature.
 */
public class PKIXValidationOptionsParser extends AbstractCustomBeanDefinitionParser {

    /** Validation Options. */
    @Nonnull public static final QName ELEMENT_NAME = new QName(SecurityNamespaceHandler.SECURITY_NAMESPACE,
            "ValidationOptions");

    /** {@inheritDoc} */
    @Override
    @Nonnull protected Class<?> getBeanClass(@Nonnull final Element element) {
        return PKIXValidationOptions.class;
    }

    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(element, parserContext, builder);

        if (element.hasAttributeNS(null, "processEmptyCRLs")) {
            builder.addPropertyValue("processEmptyCRLs",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "processEmptyCRLs")));
        }

        if (element.hasAttributeNS(null, "processExpiredCRLs")) {
            builder.addPropertyValue("processExpiredCRLs",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "processExpiredCRLs")));
        }

        if (element.hasAttributeNS(null, "processCredentialCRLs")) {
            builder.addPropertyValue("processCredentialCRLs",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "processCredentialCRLs")));
        }

        if (element.hasAttributeNS(null, "defaultVerificationDepth")) {
            builder.addPropertyValue("defaultVerificationDepth",
                    StringSupport.trimOrNull(element.getAttributeNS(null, "defaultVerificationDepth")));
        }
    }

    /** {@inheritDoc} */
    @Override protected boolean shouldGenerateId() {
        return true;
    }

}