/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.spring.testing;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.testing.OpenSAMLInitBaseTestCase;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.mock.env.MockPropertySource;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeSuite;

import net.shibboleth.shared.annotation.constraint.NonnullBeforeTest;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.spring.util.ApplicationContextBuilder;
import net.shibboleth.shared.spring.util.SpringSupport;

/**
 * Base class for testing metadata providers.
 */
public class AbstractFailFastTest extends OpenSAMLInitBaseTestCase {

    /** The directory where our test files are. */
    @NonnullBeforeTest private String workspaceDirName;

    /** All the {@link GenericApplicationContext}  ever allocated for this test. They are explicitly torn down at the end. */
    @NonnullBeforeTest static List<GenericApplicationContext> contexts;

    /** Make note of newly allocated {@link GenericApplicationContext}.
     * @param context what to remember.
     */
    protected void registerContext(@Nonnull final GenericApplicationContext context) {
        synchronized(contexts) {
            contexts.add(context);
        }        
    }

    /** Our test path.
     * @return the path
     */
    @Nonnull @NotEmpty protected String getPath() {
        return "/net/shibboleth/spring/failfast/";
    }

    /** get the directory where our tests are stored.
     * @return the directoryname
     */
    @NonnullBeforeTest protected String getWorkspaceDirName() {
        return workspaceDirName;
    }

    /** Allocate up any per test resources.
     * @throws IOException neveer
     */
    @BeforeSuite public void beforeSuite() throws IOException {
        contexts = new ArrayList<>();
    }

    /** Set up our state.
     * @throws IOException if required.
     */
    @BeforeClass public void beforeClass() throws IOException {
        final ClassPathResource resource =
                new ClassPathResource(getPath());
        workspaceDirName = resource.getFile().getAbsolutePath() + "/";
    }

    /** tear down all the {@link GenericApplicationContext} we have used. */
    @AfterSuite public void afterSuite() {
        final Iterator<GenericApplicationContext> contextIterator = contexts.iterator(); 
        while (contextIterator.hasNext()) {
            final GenericApplicationContext context;
            synchronized (contexts) {
                context = contextIterator.next();
            }
            context.close();
        }
    }

    /** Conjur up, and remember a {@link GenericApplicationContext}.
     * @param contextName what to call it
     * @param propSource Any properties to define
     * @param files all the files to read
     * @return a {@link GenericApplicationContext}
     * @throws IOException as required.
     */
    @Nonnull protected ApplicationContext getApplicationContext(@Nonnull final String contextName,
            @Nullable final MockPropertySource propSource, @Nonnull final String... files) throws IOException {
        final Resource[] resources = new Resource[files.length];

        for (int i = 0; i < files.length; i++) {
            resources[i] = new ClassPathResource(getPath() + files[i]);
        }

        final ApplicationContextBuilder builder = new ApplicationContextBuilder();
        
        builder.setName(contextName);
        
        if (propSource==null) {
            builder.setPropertySources(CollectionSupport.emptyList());
        } else {
            builder.setPropertySources(CollectionSupport.singletonList(propSource));
        }
        
        builder.setServiceConfigurations(CollectionSupport.arrayAsList(resources));

        final GenericApplicationContext context = builder.build();
        
        registerContext(context);
        
        return context;
    }

    /**
     * Conjure up, and remember a {@link GenericApplicationContext}.
     * 
     * @param contextName what to call it
     * @param files all the files to read
     * @return a {@link GenericApplicationContext} 
     * @throws IOException as required.
     */
    @Nonnull protected ApplicationContext getApplicationContext(@Nonnull final String contextName,
            @Nonnull final String... files) throws IOException {
        return getApplicationContext(contextName, null, files);
    }

    /** Read and parse some files, then look for a {@link ReloadableService} in them. Fail fast.
     * @param propSource Any properties to define
     * @param files all the files to read
     * @return the bean
     * @throws IOException as required.
     */
    @Nullable protected Object getBean(@Nullable final MockPropertySource propSource, @Nonnull final String... files)
            throws IOException {
        return getBean(propSource, true, files);
    }

    /** Read and parse some files, then look for a {@link ReloadableService} in them.
     * @param propSource Any properties to define
     * @param failFast do we fail fast or not?
     * @param files all the files to read
     * @return the bean
     * @throws IOException as required
     */
    @Nullable protected Object getBean(@Nullable final MockPropertySource propSource, @Nullable final Boolean failFast,
            @Nonnull final String... files) throws IOException {
        @SuppressWarnings("rawtypes") final Class<ReloadableService> claz = ReloadableService.class;
        
        if (propSource != null) {
            if (null == failFast) {
                propSource.setProperty("failFast", "");
            } else if (failFast) {
                propSource.setProperty("failFast", "true");
            } else {
                propSource.setProperty("failFast", "false");
            }
        }
        
        try {
            final ApplicationContext context = getApplicationContext(claz.getCanonicalName(), propSource, files);
            return SpringSupport.getBean(context, claz);
        } catch (Exception e) {
            LoggerFactory.getLogger(AbstractFailFastTest.class).debug("GetAppContext failed",e);
            return null;
        }
    }

    /** Read and parse some files, then look for a {@link ReloadableService} in them. Fail fast.
     * @param files all the files to read
     * @return the bean
     * @throws IOException as required
     */
    @Nullable protected Object getBean(@Nonnull final String... files) throws IOException {
        return getBean(null, files);
    }

    /** Create a {@link MockPropertySource} with one property in it.
     * @param name the property name
     * @param value the property value
     * @return the {@link MockPropertySource}
     */
    @Nonnull protected MockPropertySource propertySource(@Nonnull final String name, @Nonnull final String value) {
        final MockPropertySource propSource = new MockPropertySource("localProperties");
        propSource.setProperty(name, value);
        return propSource;
    }

    /** Create a {@link MockPropertySource} populated with the provided properties
     * @param values the properties (name,value) to set.
     * @return the {@link MockPropertySource}
     */
    @Nonnull protected MockPropertySource propertySource(@Nonnull final Collection<MockProperty> values) {
        final MockPropertySource propSource = new MockPropertySource("localProperties");
        for (final MockProperty property : values) {
            propSource.setProperty(property.name(), property.value());
        }
        return propSource;
    }
    
    /**
     * A record wrapper for a mock property.
     * 
     * @param name property name 
     * @param value property value
     */
    public record MockProperty(@Nonnull String name, @Nonnull String value) {
        
        /**
         * Constructor.
         *
         * @param name property name
         * @param value property value
         */
        public MockProperty(@Nonnull String name, @Nonnull String value) {
            this.name = Constraint.isNotNull(name, "Name cannot be null");
            this.value = Constraint.isNotNull(value, "Value cannot be null");
        }
        
        /**
         * Get name.
         * 
         * @return name
         */
        @Nonnull public String name() {
            assert name!=null;
            return name;
        }

        /**
         * Get value.
         * 
         * @return value
         */
        @Nonnull public String value() {
            assert value!=null;
            return value;
        }
    };
    
    /** Return the name of the file in the working dir with this name.
     * @param filePart the file name
     * @return the Fully Qualified Path
     */
    @Nonnull @NotEmpty protected String makePath(final String filePart) {
        return getWorkspaceDirName()  +  filePart;
    }

    /** Return the name of the file in the working dir with this name.
     * @param filePart the file name
     * @return the Fully Qualified Path
     */
    @Nonnull @NotEmpty protected String makeTempPath(final String filePart) {
        return getWorkspaceDirName()  +  filePart;
    }

}