/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import javax.annotation.Nonnull;

import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.primitive.NonnullSupplier;

/**
 * An implementation of {@link NonnullSupplier} which looks up the current thread-local
 * servlet request obtained from {@link HttpServletRequestResponseContext}, raising an
 * {@link IllegalStateException} if null.
 */
public class ThreadLocalHttpServletRequestSupplier implements NonnullSupplier<HttpServletRequest> {

    /**
     * {@inheritDoc}
     * Get the current HttpServletRequest from ThreadLocal storage.
     *
     * @return the current request
     */
    @Nonnull public HttpServletRequest get() {
        final HttpServletRequest req = HttpServletRequestResponseContext.getRequest();
        if (req == null) {
            throw new IllegalStateException(
                    "Current HttpServletRequest has not been loaded via HttpServletRequestResponseContext");
        }
        
        return req;
    }

}