/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.attribute.transcoding;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.saml.common.SAMLObjectBuilder;
import org.opensaml.saml.saml2.core.Attribute;
import org.opensaml.saml.saml2.metadata.RequestedAttribute;
import org.slf4j.Logger;

import com.google.common.base.Strings;

import net.shibboleth.idp.attribute.AttributeDecodingException;
import net.shibboleth.idp.attribute.AttributeEncodingException;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.IdPRequestedAttribute;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Base class for transcoders that operate on a SAML 2 {@link Attribute} or {@link RequestedAttribute}.
 * 
 * @param <EncodedType> the type of data that can be handled by the transcoder
 */
public abstract class AbstractSAML2AttributeTranscoder<EncodedType extends IdPAttributeValue> extends
        AbstractSAMLAttributeTranscoder<Attribute,EncodedType> implements SAML2AttributeTranscoder<EncodedType> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractSAML2AttributeTranscoder.class);
    
    /** Builder used to construct {@link Attribute} objects. */
    @Nonnull private final SAMLObjectBuilder<Attribute> attributeBuilder;

    /** Builder used to construct {@link RequestedAttribute} objects. */
    @Nonnull private final SAMLObjectBuilder<RequestedAttribute> reqAttributeBuilder;
    
    /** Constructor. */
    public AbstractSAML2AttributeTranscoder() {
        attributeBuilder = (SAMLObjectBuilder<Attribute>)
                XMLObjectProviderRegistrySupport.getBuilderFactory().<Attribute>ensureBuilder(
                        Attribute.TYPE_NAME);
        reqAttributeBuilder = (SAMLObjectBuilder<RequestedAttribute>)
                XMLObjectProviderRegistrySupport.getBuilderFactory().<RequestedAttribute>ensureBuilder(
                        RequestedAttribute.TYPE_NAME);
    }

    /** {@inheritDoc} */
    @Nonnull public Class<Attribute> getEncodedType() {
        return Attribute.class;
    }
    
    /** {@inheritDoc} */
    @Nullable public String getEncodedName(@Nonnull final TranscodingRule rule) {
        try {
            // SAML 2 naming should be based on only what needs to be available from the properties alone.
            return new NamingFunction().apply(buildAttribute(null, null, Attribute.class, rule,
                    CollectionSupport.emptyList()));
        } catch (final AttributeEncodingException e) {
            return null;
        }
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected Attribute buildAttribute(@Nullable final ProfileRequestContext profileRequestContext,
            @Nullable final IdPAttribute attribute, @Nonnull final Class<? extends Attribute> to,
            @Nonnull final TranscodingRule rule, @Nonnull final List<XMLObject> attributeValues)
                    throws AttributeEncodingException {

        if (attribute != null && !attribute.getValues().isEmpty() && attributeValues.isEmpty()) {
            throw new AttributeEncodingException("Failed to encode any values for attribute " + attribute.getId());
        }
        
        final Attribute samlAttribute;
        
        if (to.equals(Attribute.class)) {
            samlAttribute = attributeBuilder.buildObject();
        } else if (to.equals(RequestedAttribute.class)) {
            samlAttribute = reqAttributeBuilder.buildObject();
            if (attribute instanceof IdPRequestedAttribute) {
                ((RequestedAttribute) samlAttribute).setIsRequired(((IdPRequestedAttribute) attribute).isRequired());
            }
        } else {
            throw new AttributeEncodingException("Unsupported target object type: " + to.getName());
        }

        encodeName(profileRequestContext, attribute, samlAttribute, rule);
        
        samlAttribute.getAttributeValues().addAll(attributeValues);
        
        final String friendlyName = rule.getOrDefault(PROP_FRIENDLY_NAME, String.class,
                attribute != null ? attribute.getId() : "");
        if (friendlyName != null && !friendlyName.isBlank()) {
            samlAttribute.setFriendlyName(friendlyName);
        }
        
        return samlAttribute;
    }
    
    protected void encodeName(@Nullable final ProfileRequestContext profileRequestContext,
            @Nullable final IdPAttribute attribute, @Nonnull final Attribute samlAttribute,
            @Nonnull final TranscodingRule rule) throws AttributeEncodingException {
        
        // Use metadata tag to derive name of Attribute?
        final Boolean useMetadata = rule.getOrDefault(PROP_NAME_FROM_METADATA, Boolean.class, false);
        if (useMetadata != null && useMetadata) {
            final String id = attribute != null ? attribute.getId() :
                rule.get(AttributeTranscoderRegistry.PROP_ID, String.class);
            if (id == null) {
                log.warn("Rule specified {} but no attribute ID available", PROP_NAME_FROM_METADATA);
            } else {
                final String tagValue = getNameFromMetadata(profileRequestContext, id);
                if (tagValue != null) {
                    final int lastSpace = tagValue.lastIndexOf(' ');
                    final String name;
                    final String nameFormat;
                    if (lastSpace < 0) {
                        name = StringSupport.trimOrNull(tagValue);
                        nameFormat = null;
                    } else {
                        name = StringSupport.trimOrNull(tagValue.substring(0, lastSpace));
                        nameFormat = StringSupport.trimOrNull(tagValue.substring(lastSpace));
                    }
                    if (name != null) {
                        samlAttribute.setName(name);
                        samlAttribute.setNameFormat(nameFormat);
                        log.debug("Encoding IdPAttribute {} via metadata tag as Name {}, NameFormat {}", id,
                                name, nameFormat);
                        return;
                    }
                    log.warn("Metadata tag {}, value {}, was not in the expected form", METADATA_TAG_NAME, tagValue);
                }
            }
        }
        
        final String name = rule.get(PROP_NAME, String.class);
        if (Strings.isNullOrEmpty(name)) {
            throw new AttributeEncodingException("Required transcoder property '" + PROP_NAME + "' not found");
        }
        
        samlAttribute.setName(name);
        samlAttribute.setNameFormat(rule.getOrDefault(PROP_NAME_FORMAT, String.class, Attribute.URI_REFERENCE));
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected IdPAttribute buildIdPAttribute(
            @Nullable final ProfileRequestContext profileRequestContext, @Nonnull final Attribute attribute,
            @Nonnull final TranscodingRule rule, @Nonnull final List<IdPAttributeValue> attributeValues)
                    throws AttributeDecodingException {
        
        if (!attribute.getAttributeValues().isEmpty() && attributeValues.isEmpty()) {
            throw new AttributeDecodingException("Failed to decode any values for attribute " + attribute.getName());
        }
        
        final String id = rule.get(AttributeTranscoderRegistry.PROP_ID, String.class);
        if (Strings.isNullOrEmpty(id)) {
            throw new AttributeDecodingException("Required transcoder property 'id' not found");
        }
        assert id != null;
        
        final IdPAttribute idpAttribute;
        if (attribute instanceof RequestedAttribute reqAttribute) {
            idpAttribute = new IdPRequestedAttribute(id);
            final Boolean isRequired = reqAttribute.isRequired();
            if (isRequired != null) {
                ((IdPRequestedAttribute) idpAttribute).setRequired(isRequired);
            }
        } else {
            idpAttribute = new IdPAttribute(id);
        }
        
        idpAttribute.setValues(attributeValues);
        
        return idpAttribute;
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected Iterable<XMLObject> getValues(@Nonnull final Attribute input) {
        return input.getAttributeValues();
    }

    /**
     * A function to produce a "canonical" name for a SAML 2.0 {@link Attribute} for transcoding rules.
     */
    public static class NamingFunction implements Function<Attribute,String> {

        /** {@inheritDoc} */
        @Nullable public String apply(@Nullable final Attribute input) {
            
            if (input == null || input.getName() == null) {
                return null;
            }
            
            String format = input.getNameFormat();
            if (format == null) {
                format = Attribute.UNSPECIFIED;
            }
            
            final StringBuilder builder = new StringBuilder();
            builder.append("SAML2:{").append(format).append('}').append(input.getName());
            return builder.toString();
        }

    }

}