/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.profile.config;

import java.time.Duration;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.saml.saml2.profile.config.SAML2ProfileConfiguration;
import net.shibboleth.shared.annotation.ConfigurationSetting;
import net.shibboleth.shared.annotation.constraint.NonNegative;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;

/**
 * Interface for configuration of SAML profiles that consume assertions.
 */
public interface SAMLAssertionConsumingProfileConfiguration extends SAML2ProfileConfiguration {

    /**
     * Get the predicate used to determine if assertions must be signed.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return predicate used to determine if assertions must be signed
     */
    @ConfigurationSetting(name="requireSignedAssertions")
    boolean isRequireSignedAssertions(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get the maximum lifetime of the assertions.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return maximum lifetime of the assertions
     */
    @ConfigurationSetting(name="assertionLifetime")
    @Nullable @NonNegative Duration getAssertionLifetime(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get set of audiences to include when validating assertion, in addition to our own entityID.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return additional audiences
     */
    @Nonnull @NotLive @Unmodifiable
    @ConfigurationSetting(name="assertionAudiences")
    Set<String> getAssertionAudiences( @Nullable final ProfileRequestContext profileRequestContext);

}