/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.profile.config;

import java.time.Duration;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.shared.annotation.ConfigurationSetting;
import net.shibboleth.shared.annotation.constraint.NonNegative;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;

/**
 * Interface for SAML profiles that produce assertions.
 */
public interface SAMLAssertionProducingProfileConfiguration extends SAMLProfileConfiguration {

    /** Default assertion lifetime. */
    @Nonnull public static final Duration DEFAULT_ASSERTION_LIFETIME = Duration.ofMinutes(5);

    /**
     * Get whether generated assertions should be signed.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return whether generated assertions should be signed
     */
    @ConfigurationSetting(name="signAssertions")
    boolean isSignAssertions(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get whether to include a NotBefore attribute in the Conditions of generated assertions.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return  whether to include a NotBefore attribute in the Conditions of generated assertions
     */
    @ConfigurationSetting(name="includeConditionsNotBefore")
    boolean isIncludeConditionsNotBefore(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get the lifetime of generated assertions.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return lifetime of generated assertions
     */
    @ConfigurationSetting(name="assertionLifetime")
    @NonNegative @Nonnull Duration getAssertionLifetime(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get an unmodifiable set of audiences, in addition to the relying party(ies) to which the IdP is issuing the
     * assertion, with which an assertion may be shared.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return additional audiences to which an assertion may be shared
     */
    @Nonnull @NotLive @Unmodifiable
    @ConfigurationSetting(name="assertionAudiences")
    Set<String> getAssertionAudiences(@Nullable final ProfileRequestContext profileRequestContext);
    
}