/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.profile.config;

import java.util.function.Function;

import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.handler.MessageHandler;
import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.profile.config.ProfileConfiguration;
import net.shibboleth.shared.annotation.ConfigurationSetting;

/** Common interface for SAML profile configurations. */
public interface SAMLProfileConfiguration extends ProfileConfiguration {

    /**
     * Get the predicate used to determine if generated requests should be signed.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return predicate used to determine if generated requests should be signed
     */
    @ConfigurationSetting(name="signRequests")
    boolean isSignRequests(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get the predicate used to determine if generated responses should be signed.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return predicate used to determine if generated responses should be signed
     */
    @ConfigurationSetting(name="signResponses")
    boolean isSignResponses(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get a custom handler for a SAML message produced or consumed by this profile.
     * 
     * <p>This function MUST be stateless and reusable if statically configured, or may be stateful
     * if obtained by means of a more dynamic strategy.</p>
     * 
     * <p>The use of the {@link Function} API rather than the OpenSAML {@link MessageHandler} API is a
     * concession to making scripted or otherwise non-Java implementations easily usable, and avoiding
     * the explicit need to raise exceptions to signal errors, in cases where doing do is awkward.</p>
     * 
     * @param messageContext message context
     * 
     * @return message handler
     */
    @ConfigurationSetting(name="messageHandler")
    @Nullable Function<MessageContext,Exception> getMessageHandler(@Nullable final MessageContext messageContext);

}