/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.profile.config.logic;

import javax.annotation.Nullable;

import net.shibboleth.profile.context.RelyingPartyContext;
import net.shibboleth.profile.context.logic.AbstractRelyingPartyPredicate;
import net.shibboleth.saml.profile.config.SAMLAssertionProducingProfileConfiguration;

import org.opensaml.profile.context.ProfileRequestContext;

/**
 * A predicate that returns
 * {@link SAMLAssertionProducingProfileConfiguration#isIncludeConditionsNotBefore(ProfileRequestContext)}
 * if such a profile is available from a {@link RelyingPartyContext} obtained via a lookup function,
 * by default a child of the {@link ProfileRequestContext}.
 * 
 * <p>If unable to locate a specific setting, the predicate is true.</p>
 */
public class NotBeforeProfileConfigPredicate extends AbstractRelyingPartyPredicate {

    /** {@inheritDoc} */
    public boolean test(@Nullable final ProfileRequestContext input) {
        final RelyingPartyContext rpc = getRelyingPartyContext(input);
        if (rpc != null) {
            if (rpc.getProfileConfig() instanceof SAMLAssertionProducingProfileConfiguration downcast) {
                return downcast.isIncludeConditionsNotBefore(input);
            }
        }
        
        return true;
    }

}