/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.profile.config.navigate;

import java.util.Collection;
import java.util.Set;

import javax.annotation.Nullable;

import net.shibboleth.profile.context.RelyingPartyContext;
import net.shibboleth.profile.context.navigate.AbstractRelyingPartyLookupFunction;
import net.shibboleth.saml.profile.config.SAMLAssertionProducingProfileConfiguration;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.collect.ImmutableCollection.Builder;
import com.google.common.collect.ImmutableList;

/**
 * A function that returns the effective audience restrictions to include in assertions,
 * based on combining a relying party's entityID with the result of 
 * {@link SAMLAssertionProducingProfileConfiguration#getAssertionAudiences(ProfileRequestContext)},
 * if such a profile is available from a {@link RelyingPartyContext} obtained via a lookup function,
 * by default a child of the {@link ProfileRequestContext}.
 * 
 * <p>If a specific setting is unavailable, no values are returned.</p>
 */
public class AudienceRestrictionsLookupFunction extends AbstractRelyingPartyLookupFunction<Collection<String>> {

    /** {@inheritDoc} */
    @Nullable @NotLive @Unmodifiable public Collection<String> apply(
            @Nullable final ProfileRequestContext input) {
        final RelyingPartyContext rpc = getRelyingPartyContextLookupStrategy().apply(input);
        if (rpc != null) {
            final String id = rpc.getRelyingPartyId();
            if (rpc.getProfileConfig() instanceof SAMLAssertionProducingProfileConfiguration downcast) {
                final Set<String> additional = downcast.getAssertionAudiences(input);
                if (!additional.isEmpty()) {
                    final Builder<String> builder = ImmutableList.builder();
                    if (id != null) {
                        builder.add(rpc.getRelyingPartyId());
                    }
                    builder.addAll(additional);
                    return builder.build();
                }
            }
            
            if (id != null) {
                return CollectionSupport.singletonList(id);
            }
        }
        
        return CollectionSupport.emptyList();
    }

}