/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.saml2.profile.config;

import java.time.Duration;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.saml.profile.config.SAMLArtifactConsumerProfileConfiguration;
import net.shibboleth.saml.profile.config.SAMLAssertionConsumingProfileConfiguration;
import net.shibboleth.shared.annotation.ConfigurationSetting;
import net.shibboleth.shared.annotation.constraint.NonNegative;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.saml.saml2.core.AuthnContextComparisonTypeEnumeration;
import org.opensaml.saml.saml2.core.SubjectLocality;
import org.opensaml.saml.saml2.metadata.RequestedAttribute;

/**
 * Configuration for SAML 2.0 Browser SSO profile.
 * 
 * <p>Note that this interface extends {@link SAMLAssertionConsumingProfileConfiguration} due to both
 * IdP and SP functionality including that step, but not the "producing" interface, as that is IdP-only.</p>
 */
public interface BrowserSSOProfileConfiguration extends SAMLArtifactConsumerProfileConfiguration,
        SAML2ProfileConfiguration, SAMLAssertionConsumingProfileConfiguration {
    
    /** ID for this profile configuration. */
    @Nonnull @NotEmpty static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/saml2/sso/browser";
    
    /** Bit constant for RequestedAuthnContext feature. */
    static final int FEATURE_AUTHNCONTEXT = 0x1;

    /** Bit constant for Scoping feature. */
    static final int FEATURE_SCOPING = 0x2;

    /** Bit constant for NameIDPolicy Format feature. */
    static final int FEATURE_NAMEIDFORMAT = 0x4;

    /** Bit constant for NameIDPolicy SPNameQualifier feature. */
    static final int FEATURE_SPNAMEQUALIFIER = 0x8;

    /** Bit constant for ForceAuthn feature. */
    static final int FEATURE_FORCEAUTHN = 0x10;

    /**
     * Get whether a fresh user presence proof should be required for this request.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return true iff a fresh user presence proof should be required for this request
     */
    @ConfigurationSetting(name="forceAuthn")
    boolean isForceAuthn(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get whether the client's address must match the address in an inbound {@link SubjectLocality}
     * element during inbound SSO.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return whether to compare addresses
     */
    @ConfigurationSetting(name="checkAddress")
    boolean isCheckAddress(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get the maximum amount of time allowed to have elapsed since an incoming AuthnInstant.
     * 
     * <p>A null or 0 is interpreted as an unlimited amount.</p>
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return max time since inbound AuthnInstant
     */
    @ConfigurationSetting(name="maximumTimeSinceAuthn")
    @NonNegative @Nullable Duration getMaximumTimeSinceAuthn(
            @Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Gets the maximum number of times an assertion may be proxied to signal in the SAML request.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return maximum number of times an assertion may be proxied
     */
    @ConfigurationSetting(name="proxyCount")
    @NonNegative @Nullable Integer getProxyCount(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get the comparison operator to use when issuing SAML requests containing requested context classes.
     * 
     * <p>The actual context(s) requested is left to IdP- and SP-specific interfaces because of the differences
     * in representation.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return comparison value or null
     */
    @ConfigurationSetting(name="authnContextComparison")
    @Nullable AuthnContextComparisonTypeEnumeration getAuthnContextComparison(
            @Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Get the SPNameQualifier to include in the SAML request.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return requested SPNameQualifier
     */
    @ConfigurationSetting(name="sPNameQualifier")
    @Nullable String getSPNameQualifier(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get the AttributeConsumingServiceIndex to include in the SAML request.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return the AttributeConsumingServiceIndex
     */
    @ConfigurationSetting(name="attributeIndex")
    @Nullable Integer getAttributeIndex(@Nullable final ProfileRequestContext profileRequestContext);
    
    /**
     * Get the list of {@link RequestedAttribute} objects to include in the SAML request (via extension).
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return the requested attributes
     */
    @ConfigurationSetting(name="requestedAttributes")
    @Nonnull @Unmodifiable @NotLive Collection<RequestedAttribute> getRequestedAttributes(
            @Nullable final ProfileRequestContext profileRequestContext);
    
}