/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.saml2.profile.config;

import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.saml.profile.config.SAMLProfileConfiguration;
import net.shibboleth.shared.annotation.ConfigurationSetting;

/**
 * Base interface for SAML 2 profile configurations. 
 */
public interface SAML2ProfileConfiguration extends SAMLProfileConfiguration {

    /**
     * Gets whether to bypass verification of request signatures.
     * 
     * <p>This is typically of use to deal with broken services or to allow a
     * signer's key to be bypassed in the event that it is managed improperly.</p>
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return true iff request signatures should be ignored
     */
    @ConfigurationSetting(name="ignoreRequestSignatures")
    boolean isIgnoreRequestSignatures(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Gets whether to ignore an inability to encrypt due to external factors.
     * 
     *  <p>This allows a deployer to signal that encryption is "best effort" and
     *  can be omitted if a relying party doesn't possess a key, support a compatible
     *  algorithm, etc.</p>
     *  
     *  <p>Defaults to false.</p>
     *  
     * @param profileRequestContext current profile request context
     * 
     * @return true iff encryption should be treated as optional
     */
    @ConfigurationSetting(name="encryptionOptional")
    boolean isEncryptionOptional(@Nullable final ProfileRequestContext profileRequestContext);

    /**
     * Gets the predicate used to determine if name identifiers should be encrypted.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return predicate used to determine if name identifiers should be encrypted
     */
    @ConfigurationSetting(name="encryptNameIDs")
    boolean isEncryptNameIDs(@Nullable final ProfileRequestContext profileRequestContext);

}