/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.saml2.profile.config;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.saml.profile.config.SAMLArtifactConsumerProfileConfiguration;
import net.shibboleth.shared.annotation.ConfigurationSetting;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;

/** Configuration support for SAML 2 Single Logout. */
public interface SingleLogoutProfileConfiguration extends SAML2ProfileConfiguration,
        SAMLArtifactConsumerProfileConfiguration {

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/saml2/logout";
    
    /**
     * Get whether SOAP-based requests should be signed.
     * 
     * @param messageContext current message context
     * 
     * @return whether SOAP-based requests should be signed
     */
    boolean isSignSOAPRequests(@Nullable final MessageContext messageContext);

    /**
     * Get whether SOAP-based requests should use client TLS.
     * 
     * @param messageContext current message context
     * 
     * @return whether SOAP-based requests should use client TLS
     */
    boolean isClientTLSSOAPRequests(@Nullable final MessageContext messageContext);
    
    /**
     * Get a collection of {@link org.opensaml.saml.saml2.core.NameID} Format values for which the use of
     * the NameQualifier and SPNameQualifier attributes is defined to allow default/implicit values
     * derived from the asserting and relying parties.
     * 
     * <p>In the core standard, only the {@link org.opensaml.saml.saml2.core.NameIDType#PERSISTENT} and
     * {@link org.opensaml.saml.saml2.core.NameIDType#TRANSIENT} Formats are defined in this manner. This
     * setting identifies <strong>additional</strong> Formats that should be handled in this way.</p>
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return additional Formats for which defaulting of qualifiers is permissable
     * 
     * @since 3.4.0
     */
    @ConfigurationSetting(name="qualifiedNameIDFormats")
    @Nonnull @Unmodifiable @NotLive Collection<String> getQualifiedNameIDFormats(
            @Nullable final ProfileRequestContext profileRequestContext);
    
}