/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.saml.saml2.profile.config.navigate;

import java.time.Duration;

import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.profile.context.RelyingPartyContext;
import net.shibboleth.profile.context.navigate.AbstractRelyingPartyLookupFunction;
import net.shibboleth.saml.saml2.profile.config.BrowserSSOProfileConfiguration;

/**
 * A function that returns {@link BrowserSSOProfileConfiguration#getMaximumTimeSinceAuthn(ProfileRequestContext)}
 * if such a profile is available from a {@link RelyingPartyContext} obtained via a lookup function,
 * by default a child of the {@link ProfileRequestContext}.
 * 
 * <p>If a specific setting is unavailable, a null value is returned.</p>
 */
public class MaximumTimeSinceAuthnLookupFunction extends AbstractRelyingPartyLookupFunction<Duration> {

    /** {@inheritDoc} */
    @Nullable public Duration apply(@Nullable final ProfileRequestContext input) {
        final RelyingPartyContext rpc = getRelyingPartyContextLookupStrategy().apply(input);
        if (rpc != null) {
            if (rpc.getProfileConfig() instanceof BrowserSSOProfileConfiguration sso) {
                return sso.getMaximumTimeSinceAuthn(input);
            }
        }
        
        return null;
    }

}