/*
 * This file is part of the OWL API.
 *
 * The contents of this file are subject to the LGPL License, Version 3.0.
 *
 * Copyright (C) 2011, The University of Manchester
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 *
 * Alternatively, the contents of this file may be used under the terms of the Apache License, Version 2.0
 * in which case, the provisions of the Apache License Version 2.0 are applicable instead of those above.
 *
 * Copyright 2011, University of Manchester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.semanticweb.owlapi.vocab;

import java.net.URI;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.semanticweb.owlapi.model.IRI;

/** @author Matthew Horridge, The University Of Manchester, Bio-Health Informatics
 *         Group, Date: 26-Oct-2006 */
public enum OWLRDFVocabulary {
    // //////////////////////////////////////////////////////////////////////////////////////////////////
    //
    // OWL Vocab
    //
    // //////////////////////////////////////////////////////////////////////////////////////////////////
    /** http://www.w3.org/2002/07/owl#Thing **/
    OWL_THING(Namespaces.OWL, "Thing"),
    /** http://www.w3.org/2002/07/owl#Nothing **/
    OWL_NOTHING(Namespaces.OWL, "Nothing"),
    /** http://www.w3.org/2002/07/owl#Class **/
    OWL_CLASS(Namespaces.OWL, "Class"),
    /** http://www.w3.org/2002/07/owl#Ontology **/
    OWL_ONTOLOGY(Namespaces.OWL, "Ontology"),
    /** http://www.w3.org/2002/07/owl#OntologyProperty **/
    @Deprecated
    OWL_ONTOLOGY_PROPERTY(Namespaces.OWL, "OntologyProperty"),
    /** http://www.w3.org/2002/07/owl#imports **/
    OWL_IMPORTS(Namespaces.OWL, "imports"),
    /** http://www.w3.org/2002/07/owl#versionIRI **/
    OWL_VERSION_IRI(Namespaces.OWL, "versionIRI"),
    /** http://www.w3.org/2002/07/owl#versionInfo **/
    OWL_VERSION_INFO(Namespaces.OWL, "versionInfo"),
    /** http://www.w3.org/2002/07/owl#equivalentClass **/
    OWL_EQUIVALENT_CLASS(Namespaces.OWL, "equivalentClass"),
    /** http://www.w3.org/2002/07/owl#ObjectProperty **/
    OWL_OBJECT_PROPERTY(Namespaces.OWL, "ObjectProperty"),
    /** http://www.w3.org/2002/07/owl#DatatypeProperty **/
    OWL_DATA_PROPERTY(Namespaces.OWL, "DatatypeProperty"),
    /** http://www.w3.org/2002/07/owl#FunctionalProperty **/
    OWL_FUNCTIONAL_PROPERTY(Namespaces.OWL, "FunctionalProperty"),
    /** http://www.w3.org/2002/07/owl#InverseFunctionalProperty **/
    OWL_INVERSE_FUNCTIONAL_PROPERTY(Namespaces.OWL, "InverseFunctionalProperty"),
    /** http://www.w3.org/2002/07/owl#AntisymmetricProperty **/
    @Deprecated
    OWL_ANTI_SYMMETRIC_PROPERTY(Namespaces.OWL, "AntisymmetricProperty"),
    /** http://www.w3.org/2002/07/owl#AsymmetricProperty **/
    OWL_ASYMMETRIC_PROPERTY(Namespaces.OWL, "AsymmetricProperty"),
    /** http://www.w3.org/2002/07/owl#SymmetricProperty **/
    OWL_SYMMETRIC_PROPERTY(Namespaces.OWL, "SymmetricProperty"),
    /** http://www.w3.org/2002/07/owl#Restriction **/
    OWL_RESTRICTION(Namespaces.OWL, "Restriction"),
    /** http://www.w3.org/2002/07/owl#DataRestriction **/
    @Deprecated
    OWL_DATA_RESTRICTION(Namespaces.OWL, "DataRestriction"),
    /** http://www.w3.org/2002/07/owl#ObjectRestriction **/
    @Deprecated
    OWL_OBJECT_RESTRICTION(Namespaces.OWL, "ObjectRestriction"),
    /** http://www.w3.org/2002/07/owl#onProperty **/
    OWL_ON_PROPERTY(Namespaces.OWL, "onProperty"),
    /** http://www.w3.org/2002/07/owl#intersectionOf **/
    OWL_INTERSECTION_OF(Namespaces.OWL, "intersectionOf"),
    /** http://www.w3.org/2002/07/owl#unionOf **/
    OWL_UNION_OF(Namespaces.OWL, "unionOf"),
    /** http://www.w3.org/2002/07/owl#allValuesFrom **/
    OWL_ALL_VALUES_FROM(Namespaces.OWL, "allValuesFrom"),
    /** http://www.w3.org/2002/07/owl#someValuesFrom **/
    OWL_SOME_VALUES_FROM(Namespaces.OWL, "someValuesFrom"),
    /** http://www.w3.org/2002/07/owl#hasValue **/
    OWL_HAS_VALUE(Namespaces.OWL, "hasValue"),
    /** http://www.w3.org/2002/07/owl#disjointWith **/
    OWL_DISJOINT_WITH(Namespaces.OWL, "disjointWith"),
    /** http://www.w3.org/2002/07/owl#oneOf **/
    OWL_ONE_OF(Namespaces.OWL, "oneOf"),
    /** http://www.w3.org/2002/07/owl#SelfRestriction **/
    @Deprecated
    OWL_SELF_RESTRICTION(Namespaces.OWL, "SelfRestriction"),
    /** http://www.w3.org/2002/07/owl#hasSelf **/
    OWL_HAS_SELF(Namespaces.OWL, "hasSelf"),
    /** http://www.w3.org/2002/07/owl#disjointUnionOf **/
    OWL_DISJOINT_UNION_OF(Namespaces.OWL, "disjointUnionOf"),
    /** http://www.w3.org/2002/07/owl#minCardinality **/
    OWL_MIN_CARDINALITY(Namespaces.OWL, "minCardinality"),
    /** http://www.w3.org/2002/07/owl#minQualifiedCardinality **/
    OWL_MIN_QUALIFIED_CARDINALITY(Namespaces.OWL, "minQualifiedCardinality"),
    /** http://www.w3.org/2002/07/owl#cardinality **/
    OWL_CARDINALITY(Namespaces.OWL, "cardinality"),
    /** http://www.w3.org/2002/07/owl#qualifiedCardinality **/
    OWL_QUALIFIED_CARDINALITY(Namespaces.OWL, "qualifiedCardinality"),
    /** http://www.w3.org/2002/07/owl#AnnotationProperty **/
    OWL_ANNOTATION_PROPERTY(Namespaces.OWL, "AnnotationProperty"),
    /** http://www.w3.org/2002/07/owl#Annotation **/
    OWL_ANNOTATION(Namespaces.OWL, "Annotation"),
    /** http://www.w3.org/2002/07/owl#declaredAs **/
    @Deprecated
    OWL_DECLARED_AS(Namespaces.OWL, "declaredAs"),
    /** http://www.w3.org/2002/07/owl#Individual **/
    OWL_INDIVIDUAL(Namespaces.OWL, "Individual"),
    /** http://www.w3.org/2002/07/owl#NamedIndividual **/
    OWL_NAMED_INDIVIDUAL(Namespaces.OWL, "NamedIndividual"),
    /** http://www.w3.org/2002/07/owl#Datatype **/
    OWL_DATATYPE(Namespaces.OWL, "Datatype"),
    /** http://www.w3.org/2000/01/rdf-schema#subClassOf **/
    RDFS_SUBCLASS_OF(Namespaces.RDFS, "subClassOf"),
    /** http://www.w3.org/2000/01/rdf-schema#subPropertyOf **/
    RDFS_SUB_PROPERTY_OF(Namespaces.RDFS, "subPropertyOf"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#type **/
    RDF_TYPE(Namespaces.RDF, "type"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#nil **/
    RDF_NIL(Namespaces.RDF, "nil"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#rest **/
    RDF_REST(Namespaces.RDF, "rest"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#first **/
    RDF_FIRST(Namespaces.RDF, "first"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#List **/
    RDF_LIST(Namespaces.RDF, "List"),
    /** http://www.w3.org/2002/07/owl#maxCardinality **/
    OWL_MAX_CARDINALITY(Namespaces.OWL, "maxCardinality"),
    /** http://www.w3.org/2002/07/owl#maxQualifiedCardinality **/
    OWL_MAX_QUALIFIED_CARDINALITY(Namespaces.OWL, "maxQualifiedCardinality"),
    /** http://www.w3.org/2002/07/owl#NegativeObjectPropertyAssertion **/
    @Deprecated
    OWL_NEGATIVE_OBJECT_PROPERTY_ASSERTION(Namespaces.OWL,
            "NegativeObjectPropertyAssertion"),
    /** http://www.w3.org/2002/07/owl#NegativeDataPropertyAssertion **/
    @Deprecated
    OWL_NEGATIVE_DATA_PROPERTY_ASSERTION(Namespaces.OWL, "NegativeDataPropertyAssertion"),
    /** http://www.w3.org/2002/07/owl#NegativePropertyAssertion **/
    OWL_NEGATIVE_PROPERTY_ASSERTION(Namespaces.OWL, "NegativePropertyAssertion"),
    /** http://www.w3.org/2000/01/rdf-schema#label **/
    RDFS_LABEL(Namespaces.RDFS, "label"),
    /** http://www.w3.org/2000/01/rdf-schema#comment **/
    RDFS_COMMENT(Namespaces.RDFS, "comment"),
    /** http://www.w3.org/2000/01/rdf-schema#seeAlso **/
    RDFS_SEE_ALSO(Namespaces.RDFS, "seeAlso"),
    /** http://www.w3.org/2000/01/rdf-schema#isDefinedBy **/
    RDFS_IS_DEFINED_BY(Namespaces.RDFS, "isDefinedBy"),
    /** http://www.w3.org/2000/01/rdf-schema#Resource **/
    RDFS_RESOURCE(Namespaces.RDFS, "Resource"),
    /** http://www.w3.org/2000/01/rdf-schema#Literal **/
    RDFS_LITERAL(Namespaces.RDFS, "Literal"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#PlainLiteral **/
    RDF_PLAIN_LITERAL(Namespaces.RDF, "PlainLiteral"),
    /** http://www.w3.org/2000/01/rdf-schema#subClassOf **/
    RDFS_DATATYPE(Namespaces.RDFS, "Datatype"),
    /** http://www.w3.org/2002/07/owl#TransitiveProperty **/
    OWL_TRANSITIVE_PROPERTY(Namespaces.OWL, "TransitiveProperty"),
    /** http://www.w3.org/2002/07/owl#ReflexiveProperty **/
    OWL_REFLEXIVE_PROPERTY(Namespaces.OWL, "ReflexiveProperty"),
    /** http://www.w3.org/2002/07/owl#IrreflexiveProperty **/
    OWL_IRREFLEXIVE_PROPERTY(Namespaces.OWL, "IrreflexiveProperty"),
    /** http://www.w3.org/2002/07/owl#inverseOf **/
    OWL_INVERSE_OF(Namespaces.OWL, "inverseOf"),
    /** http://www.w3.org/2002/07/owl#complementOf **/
    OWL_COMPLEMENT_OF(Namespaces.OWL, "complementOf"),
    /** http://www.w3.org/2002/07/owl#datatypeComplementOf **/
    OWL_DATATYPE_COMPLEMENT_OF(Namespaces.OWL, "datatypeComplementOf"),
    /** http://www.w3.org/2002/07/owl#AllDifferent **/
    OWL_ALL_DIFFERENT(Namespaces.OWL, "AllDifferent"),
    /** http://www.w3.org/2002/07/owl#distinctMembers **/
    OWL_DISTINCT_MEMBERS(Namespaces.OWL, "distinctMembers"),
    /** http://www.w3.org/2002/07/owl#sameAs **/
    OWL_SAME_AS(Namespaces.OWL, "sameAs"),
    /** http://www.w3.org/2002/07/owl#differentFrom **/
    OWL_DIFFERENT_FROM(Namespaces.OWL, "differentFrom"),
    /** http://www.w3.org/2002/07/owl#DeprecatedProperty **/
    OWL_DEPRECATED_PROPERTY(Namespaces.OWL, "DeprecatedProperty"),
    /** http://www.w3.org/2002/07/owl#equivalentProperty **/
    OWL_EQUIVALENT_PROPERTY(Namespaces.OWL, "equivalentProperty"),
    /** http://www.w3.org/2002/07/owl#DeprecatedClass **/
    OWL_DEPRECATED_CLASS(Namespaces.OWL, "DeprecatedClass"),
    /** http://www.w3.org/2002/07/owl#DataRange **/
    OWL_DATA_RANGE(Namespaces.OWL, "DataRange"),
    /** http://www.w3.org/2000/01/rdf-schema#domain **/
    RDFS_DOMAIN(Namespaces.RDFS, "domain"),
    /** http://www.w3.org/2000/01/rdf-schema#range **/
    RDFS_RANGE(Namespaces.RDFS, "range"),
    /** http://www.w3.org/2000/01/rdf-schema#Class **/
    RDFS_CLASS(Namespaces.RDFS, "Class"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#Property **/
    RDF_PROPERTY(Namespaces.RDF, "Property"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#subject **/
    @Deprecated
    RDF_SUBJECT(Namespaces.RDF, "subject"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#predicate **/
    @Deprecated
    RDF_PREDICATE(Namespaces.RDF, "predicate"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#object **/
    @Deprecated
    RDF_OBJECT(Namespaces.RDF, "object"),
    /** http://www.w3.org/2002/07/owl#subject **/
    @Deprecated
    OWL_SUBJECT(Namespaces.OWL, "subject"),
    /** http://www.w3.org/2002/07/owl#predicate **/
    @Deprecated
    OWL_PREDICATE(Namespaces.OWL, "predicate"),
    /** http://www.w3.org/2002/07/owl#object **/
    @Deprecated
    OWL_OBJECT(Namespaces.OWL, "object"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#Description **/
    RDF_DESCRIPTION(Namespaces.RDF, "Description"),
    /** http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral **/
    RDF_XML_LITERAL(Namespaces.RDF, "XMLLiteral"),
    /** http://www.w3.org/2002/07/owl#priorVersion **/
    OWL_PRIOR_VERSION(Namespaces.OWL, "priorVersion"),
    /** http://www.w3.org/2002/07/owl#deprecated **/
    OWL_DEPRECATED(Namespaces.OWL, "deprecated"),
    /** http://www.w3.org/2002/07/owl#backwardCompatibleWith **/
    OWL_BACKWARD_COMPATIBLE_WITH(Namespaces.OWL, "backwardCompatibleWith"),
    /** http://www.w3.org/2002/07/owl#incompatibleWith **/
    OWL_INCOMPATIBLE_WITH(Namespaces.OWL, "incompatibleWith"),
    /** http://www.w3.org/2002/07/owl#objectPropertyDomain **/
    @Deprecated
    OWL_OBJECT_PROPERTY_DOMAIN(Namespaces.OWL, "objectPropertyDomain"),
    /** http://www.w3.org/2002/07/owl#dataPropertyDomain **/
    @Deprecated
    OWL_DATA_PROPERTY_DOMAIN(Namespaces.OWL, "dataPropertyDomain"),
    /** http://www.w3.org/2002/07/owl#dataPropertyRange **/
    @Deprecated
    OWL_DATA_PROPERTY_RANGE(Namespaces.OWL, "dataPropertyRange"),
    /** http://www.w3.org/2002/07/owl#objectPropertyRange **/
    @Deprecated
    OWL_OBJECT_PROPERTY_RANGE(Namespaces.OWL, "objectPropertyRange"),
    /** http://www.w3.org/2002/07/owl#subObjectPropertyOf **/
    @Deprecated
    OWL_SUB_OBJECT_PROPERTY_OF(Namespaces.OWL, "subObjectPropertyOf"),
    /** http://www.w3.org/2002/07/owl#subDataPropertyOf **/
    @Deprecated
    OWL_SUB_DATA_PROPERTY_OF(Namespaces.OWL, "subDataPropertyOf"),
    /** http://www.w3.org/2002/07/owl#disjointDataProperties **/
    @Deprecated
    OWL_DISJOINT_DATA_PROPERTIES(Namespaces.OWL, "disjointDataProperties"),
    /** http://www.w3.org/2002/07/owl#disjointObjectProperties **/
    @Deprecated
    OWL_DISJOINT_OBJECT_PROPERTIES(Namespaces.OWL, "disjointObjectProperties"),
    /** http://www.w3.org/2002/07/owl#propertyDisjointWith **/
    OWL_PROPERTY_DISJOINT_WITH(Namespaces.OWL, "propertyDisjointWith"),
    /** http://www.w3.org/2002/07/owl#equivalentDataProperty **/
    @Deprecated
    OWL_EQUIVALENT_DATA_PROPERTIES(Namespaces.OWL, "equivalentDataProperty"),
    /** http://www.w3.org/2002/07/owl#equivalentObjectProperty **/
    @Deprecated
    OWL_EQUIVALENT_OBJECT_PROPERTIES(Namespaces.OWL, "equivalentObjectProperty"),
    /** http://www.w3.org/2002/07/owl#FunctionalDataProperty **/
    @Deprecated
    OWL_FUNCTIONAL_DATA_PROPERTY(Namespaces.OWL, "FunctionalDataProperty"),
    /** http://www.w3.org/2002/07/owl#FunctionalObjectProperty **/
    @Deprecated
    OWL_FUNCTIONAL_OBJECT_PROPERTY(Namespaces.OWL, "FunctionalObjectProperty"),
    /** http://www.w3.org/2002/07/owl#onClass **/
    OWL_ON_CLASS(Namespaces.OWL, "onClass"),
    /** http://www.w3.org/2002/07/owl#onDataRange **/
    OWL_ON_DATA_RANGE(Namespaces.OWL, "onDataRange"),
    /** http://www.w3.org/2002/07/owl#onDatatype **/
    OWL_ON_DATA_TYPE(Namespaces.OWL, "onDatatype"),
    /** http://www.w3.org/2002/07/owl#withRestrictions **/
    OWL_WITH_RESTRICTIONS(Namespaces.OWL, "withRestrictions"),
    /** http://www.w3.org/2002/07/owl#inverseObjectPropertyExpression **/
    OWL_INVERSE_OBJECT_PROPERTY_EXPRESSION(Namespaces.OWL,
            "inverseObjectPropertyExpression"),
    /** http://www.w3.org/2002/07/owl#Axiom **/
    OWL_AXIOM(Namespaces.OWL, "Axiom"),
    /** http://www.w3.org/2002/07/owl#propertyChain **/
    @Deprecated
    OWL_PROPERTY_CHAIN(Namespaces.OWL, "propertyChain"),
    /** http://www.w3.org/2002/07/owl#propertyChainAxiom **/
    OWL_PROPERTY_CHAIN_AXIOM(Namespaces.OWL, "propertyChainAxiom"),
    /** http://www.w3.org/2002/07/owl#AllDisjointClasses **/
    OWL_ALL_DISJOINT_CLASSES(Namespaces.OWL, "AllDisjointClasses"),
    /** http://www.w3.org/2002/07/owl#members **/
    OWL_MEMBERS(Namespaces.OWL, "members"),
    /** http://www.w3.org/2002/07/owl#AllDisjointProperties **/
    OWL_ALL_DISJOINT_PROPERTIES(Namespaces.OWL, "AllDisjointProperties"),
    /** http://www.w3.org/2002/07/owl#topObjectProperty **/
    OWL_TOP_OBJECT_PROPERTY(Namespaces.OWL, "topObjectProperty"),
    /** http://www.w3.org/2002/07/owl#bottomObjectProperty **/
    OWL_BOTTOM_OBJECT_PROPERTY(Namespaces.OWL, "bottomObjectProperty"),
    /** http://www.w3.org/2002/07/owl#topDataProperty **/
    OWL_TOP_DATA_PROPERTY(Namespaces.OWL, "topDataProperty"),
    /** http://www.w3.org/2002/07/owl#bottomDataProperty **/
    OWL_BOTTOM_DATA_PROPERTY(Namespaces.OWL, "bottomDataProperty"),
    /** http://www.w3.org/2002/07/owl#hasKey **/
    OWL_HAS_KEY(Namespaces.OWL, "hasKey"),
    /** http://www.w3.org/2002/07/owl#annotatedSource **/
    OWL_ANNOTATED_SOURCE(Namespaces.OWL, "annotatedSource"),
    /** http://www.w3.org/2002/07/owl#annotatedProperty **/
    OWL_ANNOTATED_PROPERTY(Namespaces.OWL, "annotatedProperty"),
    /** http://www.w3.org/2002/07/owl#annotatedTarget **/
    OWL_ANNOTATED_TARGET(Namespaces.OWL, "annotatedTarget"),
    /** http://www.w3.org/2002/07/owl#sourceIndividual **/
    OWL_SOURCE_INDIVIDUAL(Namespaces.OWL, "sourceIndividual"),
    /** http://www.w3.org/2002/07/owl#assertionProperty **/
    OWL_ASSERTION_PROPERTY(Namespaces.OWL, "assertionProperty"),
    /** http://www.w3.org/2002/07/owl#targetIndividual **/
    OWL_TARGET_INDIVIDUAL(Namespaces.OWL, "targetIndividual"),
    /** http://www.w3.org/2002/07/owl#targetValue **/
    OWL_TARGET_VALUE(Namespaces.OWL, "targetValue");
    final IRI iri;
    final Namespaces namespace;
    final String shortName;

    OWLRDFVocabulary(Namespaces namespace, String shortName) {
        this.namespace = namespace;
        this.shortName = shortName;
        iri = IRI.create(namespace.toString(), shortName);
    }

    /** @return The URI
     * @deprecated Use getIRI() instead. */
    @Deprecated
    public URI getURI() {
        return iri.toURI();
    }

    /** @return the entry IRI */
    public IRI getIRI() {
        return iri;
    }

    /** @return the entry namespace */
    public Namespaces getNamespace() {
        return namespace;
    }

    /** @return the IRI fragment */
    public String getShortName() {
        return shortName;
    }

    /** Set of all IRIs for this enum values */
    public static final Set<IRI> BUILT_IN_VOCABULARY_IRIS;
    static {
        Set<IRI> set = new HashSet<IRI>();
        for (OWLRDFVocabulary v : OWLRDFVocabulary.values()) {
            set.add(v.getIRI());
        }
        BUILT_IN_VOCABULARY_IRIS = Collections.unmodifiableSet(set);
    }
    /** label , comment , versionInfo , backwardCompatibleWith , priorVersion ,
     * seeAlso , isDefinedBy , incompatibleWith , deprecated */
    public static final Set<IRI> BUILT_IN_ANNOTATION_PROPERTY_IRIS;
    static {
        BUILT_IN_ANNOTATION_PROPERTY_IRIS = new HashSet<IRI>();
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(RDFS_LABEL.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(RDFS_COMMENT.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(OWL_VERSION_INFO.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(OWL_BACKWARD_COMPATIBLE_WITH.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(OWL_PRIOR_VERSION.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(RDFS_SEE_ALSO.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(RDFS_IS_DEFINED_BY.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(OWL_INCOMPATIBLE_WITH.getIRI());
        BUILT_IN_ANNOTATION_PROPERTY_IRIS.add(OWL_DEPRECATED.getIRI());
    }

    @Override
    public String toString() {
        return iri.toString();
    }
}
