/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.aws;

import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.jackrabbit.oak.segment.aws.AwsSegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.aws.AwsSegmentArchiveWriter;
import org.apache.jackrabbit.oak.segment.aws.S3Directory;
import org.apache.jackrabbit.oak.segment.remote.RemoteUtilities;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveManager;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AwsArchiveManager
implements SegmentArchiveManager {
    private static final Logger log = LoggerFactory.getLogger(AwsArchiveManager.class);
    private static final String SEGMENT_FILE_NAME_PATTERN = "^([0-9a-f]{4})\\.([0-9a-f-]+)$";
    private final S3Directory directory;
    private final IOMonitor ioMonitor;
    private final FileStoreMonitor monitor;

    public AwsArchiveManager(S3Directory directory, IOMonitor ioMonitor, FileStoreMonitor fileStoreMonitor) {
        this.directory = directory;
        this.ioMonitor = ioMonitor;
        this.monitor = fileStoreMonitor;
    }

    public List<String> listArchives() throws IOException {
        List<String> archiveNames = this.directory.listPrefixes().stream().filter(i -> i.endsWith(".tar/")).map(x$0 -> Paths.get(x$0, new String[0])).map(Path::getFileName).map(Path::toString).collect(Collectors.toList());
        Iterator it = archiveNames.iterator();
        while (it.hasNext()) {
            String archiveName = (String)it.next();
            if (!this.isArchiveEmpty(archiveName)) continue;
            this.delete(archiveName);
            it.remove();
        }
        return archiveNames;
    }

    private boolean isArchiveEmpty(String archiveName) throws IOException {
        return this.directory.withDirectory(archiveName).listObjects("0000.").isEmpty();
    }

    public SegmentArchiveReader open(String archiveName) throws IOException {
        S3Directory archiveDirectory = this.directory.withDirectory(archiveName);
        if (!archiveDirectory.doesObjectExist("closed")) {
            throw new IOException("The archive " + archiveName + " hasn't been closed correctly.");
        }
        return new AwsSegmentArchiveReader(archiveDirectory, archiveName, this.ioMonitor);
    }

    public SegmentArchiveReader forceOpen(String archiveName) throws IOException {
        S3Directory archiveDirectory = this.directory.withDirectory(archiveName);
        return new AwsSegmentArchiveReader(archiveDirectory, archiveName, this.ioMonitor);
    }

    public SegmentArchiveWriter create(String archiveName) throws IOException {
        return new AwsSegmentArchiveWriter(this.directory.withDirectory(archiveName), archiveName, this.ioMonitor, this.monitor);
    }

    public boolean delete(String archiveName) {
        return this.directory.withDirectory(archiveName).deleteAllObjects();
    }

    public boolean renameTo(String from, String to) {
        try {
            S3Directory fromDirectory = this.directory.withDirectory(from);
            S3Directory toDirectory = this.directory.withDirectory(to);
            for (S3ObjectSummary obj : fromDirectory.listObjects("")) {
                toDirectory.copyObject(fromDirectory, obj.getKey());
            }
            fromDirectory.deleteAllObjects();
            return true;
        }
        catch (IOException e) {
            log.error("Can't rename archive {} to {}", new Object[]{from, to, e});
            return false;
        }
    }

    public void copyFile(String from, String to) throws IOException {
        S3Directory fromDirectory = this.directory.withDirectory(from);
        fromDirectory.listObjects("").forEach(obj -> {
            try {
                this.directory.withDirectory(to).copyObject(fromDirectory, obj.getKey());
            }
            catch (IOException e) {
                log.error("Can't copy segment {}", (Object)obj.getKey(), (Object)e);
            }
        });
    }

    public boolean exists(String archiveName) {
        try {
            return this.directory.withDirectory(archiveName).listObjects("").size() > 0;
        }
        catch (IOException e) {
            log.error("Can't check the existence of {}", (Object)archiveName, (Object)e);
            return false;
        }
    }

    public void recoverEntries(String archiveName, LinkedHashMap<UUID, byte[]> entries) throws IOException {
        Pattern pattern = Pattern.compile(SEGMENT_FILE_NAME_PATTERN);
        ArrayList<RecoveredEntry> entryList = new ArrayList<RecoveredEntry>();
        for (S3ObjectSummary b : this.directory.withDirectory(archiveName).listObjects("")) {
            String name = Paths.get(b.getKey(), new String[0]).getFileName().toString();
            Matcher m = pattern.matcher(name);
            if (!m.matches()) continue;
            int position = Integer.parseInt(m.group(1), 16);
            UUID uuid = UUID.fromString(m.group(2));
            byte[] data = this.directory.readObject(b.getKey());
            entryList.add(new RecoveredEntry(position, uuid, data, name));
        }
        Collections.sort(entryList);
        int i = 0;
        for (RecoveredEntry e : entryList) {
            if (e.position != i) {
                log.warn("Missing entry {}.??? when recovering {}. No more segments will be read.", (Object)String.format("%04X", i), (Object)archiveName);
                break;
            }
            log.info("Recovering segment {}/{}", (Object)archiveName, (Object)e.fileName);
            entries.put(e.uuid, e.data);
            ++i;
        }
    }

    public void backup(String archiveName, String backupArchiveName, Set<UUID> recoveredEntries) throws IOException {
        this.copyFile(archiveName, backupArchiveName);
        this.delete(archiveName, recoveredEntries);
    }

    private void delete(String archiveName, Set<UUID> recoveredEntries) throws IOException {
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        for (S3ObjectSummary b : this.directory.withDirectory(archiveName).listObjects("")) {
            String name = Paths.get(b.getKey(), new String[0]).getFileName().toString();
            UUID uuid = RemoteUtilities.getSegmentUUID((String)name);
            if (recoveredEntries.contains(uuid)) continue;
            keys.add(new DeleteObjectsRequest.KeyVersion(b.getKey()));
        }
        this.directory.withDirectory(archiveName).deleteObjects(keys);
    }

    private static class RecoveredEntry
    implements Comparable<RecoveredEntry> {
        private final byte[] data;
        private final UUID uuid;
        private final int position;
        private final String fileName;

        public RecoveredEntry(int position, UUID uuid, byte[] data, String fileName) {
            this.data = data;
            this.uuid = uuid;
            this.position = position;
            this.fileName = fileName;
        }

        @Override
        public int compareTo(RecoveredEntry o) {
            return Integer.compare(this.position, o.position);
        }
    }
}

