/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.easymock.EasyMock;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class FileRecordsTest {
    private byte[][] values = new byte[][]{"abcd".getBytes(), "efgh".getBytes(), "ijkl".getBytes()};
    private FileRecords fileRecords;

    @Before
    public void setup() throws IOException {
        this.fileRecords = this.createFileRecords(this.values);
    }

    @Test
    public void testFileSize() throws IOException {
        Assert.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        for (int i = 0; i < 20; ++i) {
            this.fileRecords.append(MemoryRecords.withRecords((CompressionType)CompressionType.NONE, (SimpleRecord[])new SimpleRecord[]{new SimpleRecord("abcd".getBytes())}));
            Assert.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        }
    }

    @Test
    public void testIterationOverPartialAndTruncation() throws IOException {
        this.testPartialWrite(0, this.fileRecords);
        this.testPartialWrite(2, this.fileRecords);
        this.testPartialWrite(4, this.fileRecords);
        this.testPartialWrite(5, this.fileRecords);
        this.testPartialWrite(6, this.fileRecords);
    }

    private void testPartialWrite(int size, FileRecords fileRecords) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(size);
        for (int i = 0; i < size; ++i) {
            buffer.put((byte)0);
        }
        buffer.rewind();
        fileRecords.channel().write(buffer);
        Iterator records = fileRecords.records().iterator();
        for (byte[] value : this.values) {
            Assert.assertTrue((boolean)records.hasNext());
            Assert.assertEquals((Object)((Record)records.next()).value(), (Object)ByteBuffer.wrap(value));
        }
    }

    @Test
    public void testIterationDoesntChangePosition() throws IOException {
        long position = this.fileRecords.channel().position();
        Iterator records = this.fileRecords.records().iterator();
        for (byte[] value : this.values) {
            Assert.assertTrue((boolean)records.hasNext());
            Assert.assertEquals((Object)((Record)records.next()).value(), (Object)ByteBuffer.wrap(value));
        }
        Assert.assertEquals((long)position, (long)this.fileRecords.channel().position());
    }

    @Test
    public void testRead() throws IOException {
        FileRecords read = this.fileRecords.read(0, this.fileRecords.sizeInBytes());
        TestUtils.checkEquals(this.fileRecords.batches(), read.batches());
        List<RecordBatch> items = FileRecordsTest.batches((Records)read);
        RecordBatch second = items.get(1);
        read = this.fileRecords.read(second.sizeInBytes(), this.fileRecords.sizeInBytes());
        Assert.assertEquals((String)"Try a read starting from the second message", items.subList(1, 3), FileRecordsTest.batches((Records)read));
        read = this.fileRecords.read(second.sizeInBytes(), second.sizeInBytes());
        Assert.assertEquals((String)"Try a read of a single message starting from the second message", Collections.singletonList(second), FileRecordsTest.batches((Records)read));
    }

    @Test
    public void testSearch() throws IOException {
        SimpleRecord lastMessage = new SimpleRecord("test".getBytes());
        this.fileRecords.append(MemoryRecords.withRecords((long)50L, (CompressionType)CompressionType.NONE, (SimpleRecord[])new SimpleRecord[]{lastMessage}));
        List<RecordBatch> batches = FileRecordsTest.batches((Records)this.fileRecords);
        int position = 0;
        int message1Size = batches.get(0).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find the first message by its offset", (Object)new FileRecords.LogOffsetPosition(0L, position, message1Size), (Object)this.fileRecords.searchForOffsetWithSize(0L, 0));
        int message2Size = batches.get(1).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find second message when starting from 0", (Object)new FileRecords.LogOffsetPosition(1L, position += message1Size, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, 0));
        Assert.assertEquals((String)"Should be able to find second message starting from its offset", (Object)new FileRecords.LogOffsetPosition(1L, position, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, position));
        int message4Size = batches.get(3).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find fourth message from a non-existant offset", (Object)new FileRecords.LogOffsetPosition(50L, position += message2Size + batches.get(2).sizeInBytes(), message4Size), (Object)this.fileRecords.searchForOffsetWithSize(3L, position));
        Assert.assertEquals((String)"Should be able to find fourth message by correct offset", (Object)new FileRecords.LogOffsetPosition(50L, position, message4Size), (Object)this.fileRecords.searchForOffsetWithSize(50L, position));
    }

    @Test
    public void testIteratorWithLimits() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = batch.sizeInBytes();
        FileRecords slice = this.fileRecords.read(start, size);
        Assert.assertEquals(Collections.singletonList(batch), FileRecordsTest.batches((Records)slice));
        FileRecords slice2 = this.fileRecords.read(start, size - 1);
        Assert.assertEquals(Collections.emptyList(), FileRecordsTest.batches((Records)slice2));
    }

    @Test
    public void testTruncate() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(0);
        int end = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        this.fileRecords.truncateTo(end);
        Assert.assertEquals(Collections.singletonList(batch), FileRecordsTest.batches((Records)this.fileRecords));
        Assert.assertEquals((long)batch.sizeInBytes(), (long)this.fileRecords.sizeInBytes());
    }

    @Test
    public void testTruncateNotCalledIfSizeIsSameAsTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null);
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(42);
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testTruncateNotCalledIfSizeIsBiggerThanTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null);
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        try {
            fileRecords.truncateTo(43);
            Assert.fail((String)"Should throw KafkaException");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testTruncateIfSizeIsDifferentToTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null).once();
        EasyMock.expect((Object)channelMock.truncate(23L)).andReturn(null).once();
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(23);
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testPreallocateTrue() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)true);
        long position = fileRecords.channel().position();
        int size = fileRecords.sizeInBytes();
        Assert.assertEquals((long)0L, (long)position);
        Assert.assertEquals((long)0L, (long)size);
        Assert.assertEquals((long)0x20000000L, (long)temp.length());
    }

    @Test
    public void testPreallocateFalse() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords set = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)false);
        long position = set.channel().position();
        int size = set.sizeInBytes();
        Assert.assertEquals((long)0L, (long)position);
        Assert.assertEquals((long)0L, (long)size);
        Assert.assertEquals((long)0L, (long)temp.length());
    }

    @Test
    public void testPreallocateClearShutdown() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)true);
        this.append(fileRecords, this.values);
        int oldPosition = (int)fileRecords.channel().position();
        int oldSize = fileRecords.sizeInBytes();
        Assert.assertEquals((long)this.fileRecords.sizeInBytes(), (long)oldPosition);
        Assert.assertEquals((long)this.fileRecords.sizeInBytes(), (long)oldSize);
        fileRecords.close();
        File tempReopen = new File(temp.getAbsolutePath());
        FileRecords setReopen = FileRecords.open((File)tempReopen, (boolean)true, (int)0x20000000, (boolean)true);
        int position = (int)setReopen.channel().position();
        int size = setReopen.sizeInBytes();
        Assert.assertEquals((long)oldPosition, (long)position);
        Assert.assertEquals((long)oldPosition, (long)size);
        Assert.assertEquals((long)oldPosition, (long)tempReopen.length());
    }

    @Test
    public void testFormatConversionWithPartialMessage() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = batch.sizeInBytes();
        FileRecords slice = this.fileRecords.read(start, size - 1);
        Records messageV0 = slice.downConvert((byte)0, 0L);
        Assert.assertTrue((String)"No message should be there", (boolean)FileRecordsTest.batches(messageV0).isEmpty());
        Assert.assertEquals((String)("There should be " + (size - 1) + " bytes"), (long)(size - 1), (long)messageV0.sizeInBytes());
    }

    @Test
    public void testConversion() throws IOException {
        this.doTestConversion(CompressionType.NONE, (byte)0);
        this.doTestConversion(CompressionType.GZIP, (byte)0);
        this.doTestConversion(CompressionType.NONE, (byte)1);
        this.doTestConversion(CompressionType.GZIP, (byte)1);
        this.doTestConversion(CompressionType.NONE, (byte)2);
        this.doTestConversion(CompressionType.GZIP, (byte)2);
    }

    private void doTestConversion(CompressionType compressionType, byte toMagic) throws IOException {
        int i;
        List<Long> offsets = Arrays.asList(0L, 2L, 3L, 9L, 11L, 15L, 16L, 17L, 22L, 24L);
        List<SimpleRecord> records = Arrays.asList(new SimpleRecord(1L, "k1".getBytes(), "hello".getBytes()), new SimpleRecord(2L, "k2".getBytes(), "goodbye".getBytes()), new SimpleRecord(3L, "k3".getBytes(), "hello again".getBytes()), new SimpleRecord(4L, "k4".getBytes(), "goodbye for now".getBytes()), new SimpleRecord(5L, "k5".getBytes(), "hello again".getBytes()), new SimpleRecord(6L, "k6".getBytes(), "I sense indecision".getBytes()), new SimpleRecord(7L, "k7".getBytes(), "what now".getBytes()), new SimpleRecord(8L, "k8".getBytes(), "running out".getBytes()), new SimpleRecord(9L, "k9".getBytes(), "ok, almost done".getBytes()), new SimpleRecord(10L, "k10".getBytes(), "finally".getBytes()));
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)0, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 0; i < 3; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)1, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 3; i < 6; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 6; i < 10; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        buffer.flip();
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(MemoryRecords.readableRecords((ByteBuffer)buffer));
            fileRecords.flush();
            Records convertedRecords = fileRecords.downConvert(toMagic, 0L);
            this.verifyConvertedRecords(records, offsets, convertedRecords, compressionType, toMagic);
            if (toMagic <= 1 && compressionType == CompressionType.NONE) {
                long firstOffset = toMagic == 0 ? 11L : 17L;
                Records convertedRecords2 = fileRecords.downConvert(toMagic, firstOffset);
                ArrayList<Long> filteredOffsets = new ArrayList<Long>(offsets);
                ArrayList<SimpleRecord> filteredRecords = new ArrayList<SimpleRecord>(records);
                int index = filteredOffsets.indexOf(firstOffset) - 1;
                filteredRecords.remove(index);
                filteredOffsets.remove(index);
                this.verifyConvertedRecords(filteredRecords, filteredOffsets, convertedRecords2, compressionType, toMagic);
            } else {
                Records convertedRecords2 = fileRecords.downConvert(toMagic, 10L);
                this.verifyConvertedRecords(records, offsets, convertedRecords2, compressionType, toMagic);
            }
        }
    }

    private String utf8(ByteBuffer buffer) {
        return Utils.utf8((ByteBuffer)buffer, (int)buffer.remaining());
    }

    private void verifyConvertedRecords(List<SimpleRecord> initialRecords, List<Long> initialOffsets, Records convertedRecords, CompressionType compressionType, byte magicByte) {
        int i = 0;
        for (RecordBatch batch : convertedRecords.batches()) {
            Assert.assertTrue((String)("Magic byte should be lower than or equal to " + magicByte), (batch.magic() <= magicByte ? 1 : 0) != 0);
            if (batch.magic() == 0) {
                Assert.assertEquals((Object)TimestampType.NO_TIMESTAMP_TYPE, (Object)batch.timestampType());
            } else {
                Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)batch.timestampType());
            }
            Assert.assertEquals((String)"Compression type should not be affected by conversion", (Object)compressionType, (Object)batch.compressionType());
            for (Record record : batch) {
                Assert.assertTrue((String)("Inner record should have magic " + magicByte), (boolean)record.hasMagic(batch.magic()));
                Assert.assertEquals((String)"Offset should not change", (long)initialOffsets.get(i), (long)record.offset());
                Assert.assertEquals((String)"Key should not change", (Object)this.utf8(initialRecords.get(i).key()), (Object)this.utf8(record.key()));
                Assert.assertEquals((String)"Value should not change", (Object)this.utf8(initialRecords.get(i).value()), (Object)this.utf8(record.value()));
                Assert.assertFalse((boolean)record.hasTimestampType(TimestampType.LOG_APPEND_TIME));
                if (batch.magic() == 0) {
                    Assert.assertEquals((long)-1L, (long)record.timestamp());
                    Assert.assertFalse((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                    Assert.assertTrue((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                } else if (batch.magic() == 1) {
                    Assert.assertEquals((String)"Timestamp should not change", (long)initialRecords.get(i).timestamp(), (long)record.timestamp());
                    Assert.assertTrue((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                    Assert.assertFalse((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                } else {
                    Assert.assertEquals((String)"Timestamp should not change", (long)initialRecords.get(i).timestamp(), (long)record.timestamp());
                    Assert.assertFalse((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                    Assert.assertFalse((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                }
                ++i;
            }
        }
        Assert.assertEquals((long)initialOffsets.size(), (long)i);
    }

    private static List<RecordBatch> batches(Records buffer) {
        return TestUtils.toList(buffer.batches());
    }

    private FileRecords createFileRecords(byte[][] values) throws IOException {
        FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());
        this.append(fileRecords, values);
        return fileRecords;
    }

    private void append(FileRecords fileRecords, byte[][] values) throws IOException {
        long offset = 0L;
        for (byte[] value : values) {
            ByteBuffer buffer = ByteBuffer.allocate(128);
            MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)CompressionType.NONE, (TimestampType)TimestampType.CREATE_TIME, (long)offset);
            builder.appendWithOffset(offset++, System.currentTimeMillis(), null, value);
            fileRecords.append(builder.build());
        }
        fileRecords.flush();
    }
}

