/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package aQute.lib.osgi;


import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.jar.Attributes.Name;


/**
 * Extension of {@link aQute.lib.osgi.Builder} to handle nuxeo behaviors
 *
 * @author <a href="mailto:dev@felix.apache.org">Felix Project Team</a>
 */
public final class NuxeoAnalyzer extends Analyzer
{
    private static final Name bUNDLEVENDORNAME =   new Name(BUNDLE_VENDOR);
    private static final Name nUXEOREQUIRENAME = new Name("Nuxeo-Require");
    private static final Name rEQUIREBUNDLENAME = new Name(REQUIRE_BUNDLE);
    private static final Name eXPORTPACKAGENAME = new Name(EXPORT_PACKAGE);
    private static final Name iMPORTPACKAGENAME = new Name(EXPORT_PACKAGE);

    public void addExportsVersion() throws Exception {
        Manifest manifest = dot.getManifest();
        // Patch export clauses with version
        Attributes main = manifest.getMainAttributes();
        String version = getReplacer().process(main.getValue(BUNDLE_VERSION));
        Iterator it = exports.values().iterator();
        while(it.hasNext()) {
            Map directives = (Map)it.next();
            directives.put(VERSION_ATTRIBUTE,version);
        }
        String clauses = printClauses(exports,
                "version:|uses:|include:|exclude:|mandatory:|" + IMPORT_DIRECTIVE, true);
        main.putValue(EXPORT_PACKAGE, clauses);
    }
    /*
     * Remove META-INF subfolders from exports and set package versions to bundle version.
     */
    public Map analyzeBundleClasspath(  )
        throws IOException
    {

        String bundleVersion = getProperty( BUNDLE_VERSION );
        for ( Iterator it = contained.entrySet().iterator(); it.hasNext(); )
        {
            Map.Entry entry = ( Map.Entry ) it.next();

            /* remove packages under META-INF */
            String packageName = ( String ) entry.getKey();
            if ( packageName.startsWith( "META-INF." ) )
            {
                it.remove();
            }

            /* set package versions to bundle version values */
            if ( bundleVersion != null )
            {
                Map values = ( Map ) entry.getValue();
                if ( values.get( "version" ) == null )
                {
                    values.put( "version", bundleVersion );
                }
            }
        }
        return null;
    }

    public List<String> inventoryPackagesFromContent(Jar bundle) {
       List<String> pkgs = new ArrayList<String>();
       Map<String, Map<String, Resource>> map = bundle.getDirectories();
        for (Iterator<String> i = map.keySet().iterator(); i.hasNext();) {
            String directory = (String) i.next();
            // TODO replace by a filter
            if (directory.equals("META-INF")
                    || directory.startsWith("META-INF/"))
                continue;
           if (directory.equals("OSGI-INF")
                    || directory.startsWith("OSGI-INF/"))
                continue;
            if (directory.equals("OSGI-OPT")
                    || directory.startsWith("OSGI-OPT/"))
                continue;
            if (directory.equals("/"))
                continue;
            if (directory.endsWith("/"))
                directory = directory.substring(0, directory.length() - 1);
            pkgs.add(directory.replace('/', '.'));
        }
        return pkgs;
    }
    @Override
    public String calculateExportsFromContents(Jar bundle) {
        String ddel = "";
        StringBuffer sb = new StringBuffer();
        for (String pkg:inventoryPackagesFromContent(bundle)) {
            sb.append(ddel);
            sb.append(pkg);
            ddel = ",";
        }
        return sb.toString();
    }

    @Override
    Map<String, Map<String, String>> addExportsToImports(
            Map<String, Map<String, String>> exports) {
        return newHashMap();
    }


    @Override
    public void mergeManifest(Manifest manifest) throws IOException {
        Attributes attributes = manifest.getMainAttributes();

        if (isNuxeo( attributes)) {
            filterAttributes(attributes);
        }
        super.mergeManifest(manifest);
    }

    protected boolean isNuxeo( Attributes attributes) throws IOException {
        if (attributes.containsKey(bUNDLEVENDORNAME)) {
            String vendor = attributes.getValue(bUNDLEVENDORNAME);
            if (!vendor.toLowerCase().contains("nuxeo")) {
                return false;
            }
        }
        return true;
    }

    protected void filterAttributes(Attributes attributes) {
        if (attributes.containsKey(rEQUIREBUNDLENAME)) {
            attributes.put(nUXEOREQUIRENAME, attributes.get(rEQUIREBUNDLENAME));
            attributes.remove(rEQUIREBUNDLENAME);
        }
        if (attributes.containsKey(eXPORTPACKAGENAME)) {
            attributes.remove(eXPORTPACKAGENAME);
        }
        if (attributes.containsKey(iMPORTPACKAGENAME)) {
            attributes.remove(iMPORTPACKAGENAME);
        }
    }

    @Override
    void merge(Manifest result, Manifest old) throws IOException {
        Attributes oldAttributes = old.getMainAttributes();
        if (isNuxeo(oldAttributes)) {
            filterAttributes(oldAttributes);
        }
        super.merge(result, old);
    }

}
