package org.apache.maven.plugin.eclipse;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.jar.Attributes.Name;

import org.apache.commons.io.filefilter.NameFileFilter;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.eclipse.writers.EclipseWriterConfig;
import org.apache.maven.plugin.ide.IdeDependency;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.shared.osgi.Maven2OsgiConverter;

import aQute.lib.osgi.Analyzer;
import aQute.lib.osgi.Builder;
import aQute.lib.osgi.Jar;
import aQute.lib.osgi.NuxeoAnalyzer;

public class OSGIfier {

    Log log;

    File ws;

    File dir;

    MavenProjectBuilder builder;

    Maven2OsgiConverter converter;

    File defaultDirectives;

    public OSGIfier(Log log, File dir, MavenProjectBuilder builder, Maven2OsgiConverter converter) {
        super();
        this.dir = dir;
        this.log = log;
        this.builder = builder;
        this.converter = converter;
    }

    public static File getDirectives(File dir, String filename) {
        if (dir == null) {
            return null;
        }
            File[] files;
            try {
                files = dir.listFiles((FileFilter) new NameFileFilter(filename));
            } catch (Throwable e) {
                throw new Error("Cannot list bndl dir " + dir.getPath(), e);
            }
            if (files == null || files.length == 0) {
                return getDirectives(dir.getParentFile(), filename);
            }
            return files[0];
    }


    public File getDirectives(String filename) {
        return getDirectives(dir, filename);
    }
    public File getDirectives(Artifact artifact) {
        return getDirectives(dir, converter.getBundleFileName(artifact).replace(".jar", ".bnd"));
    }

    public void addDirectives(NuxeoAnalyzer analyzer, Artifact artifact, boolean optionalResolution) throws IOException {
        analyzer.setPedantic(true);
        Jar dot = analyzer.getJar();
        // TODO provide a way to contribute by parent pom to default configuration
        analyzer.setProperty(Analyzer.IMPORT_PACKAGE, "*");
        if (!dot.getManifest().getMainAttributes().containsKey(new Name(Analyzer.BUNDLE_SYMBOLICNAME))) {
            analyzer.setProperty(Analyzer.BUNDLE_SYMBOLICNAME, converter.getBundleSymbolicName(artifact) + ";singleton:=true");	//
        }
        analyzer.setProperty(Analyzer.EXPORT_PACKAGE, analyzer.calculateExportsFromContents(dot));
        analyzer.setProperty(Analyzer.BUNDLE_CLASSPATH, ".");
        analyzer.setProperty(Analyzer.BUNDLE_ACTIVATIONPOLICY, "lazy");
        analyzer.setProperty(Analyzer.BUNDLE_VERSION, analyzer.getReplacer().process(converter.getVersion(artifact.getBaseVersion())));
        analyzer.setProperty(Analyzer.BUNDLE_REQUIREDEXECUTIONENVIRONMENT, "JavaSE-1.6");
        addDirectives(analyzer, getDirectives("default.bnd"));
        addDirectives(analyzer, getDirectives(artifact));
    }

    public void addDirectives(Analyzer analyzer, File file) {
        if (file == null) {
            return;
        }
        try {
            analyzer.addProperties(file);
        } catch (Exception e) {
            throw new Error("Cannot load directives from " + file.getPath(), e);
        }
        log.info("Added directives  " + file.getPath());
    }

    public void osgify(Artifact artifact) throws Exception {

        NuxeoAnalyzer analyzer = new NuxeoAnalyzer();
        try {

            analyzer.setJar(artifact.getFile());

            addDirectives(analyzer, defaultDirectives);
            addDirectives(analyzer, artifact, true);

            Jar dot = analyzer.getJar();

            analyzer.mergeManifest(dot.getManifest());

            // version
            String version = analyzer.getProperty(Analyzer.BUNDLE_VERSION);
            if (version != null) {
                version = Builder.cleanupVersion(version);
                analyzer.setProperty(Analyzer.BUNDLE_VERSION, version);
            }


            analyzer.calcManifest();
            analyzer.addExportsVersion();
            Jar jar = analyzer.getJar();
            File output = File.createTempFile("maven-bundle-", ".jar");
            output.deleteOnExit();
            jar.write(output);
            jar.close();
            artifact.setFile(output);
        } finally {
            analyzer.close();
        }
    }

    @SuppressWarnings("unchecked")
    protected File[] getClasspath(EclipseWriterConfig config) {
        final IdeDependency[] deps = config.getDeps();
        List<File> cp = new ArrayList(deps.length);
        for (int i = 0; i < deps.length; ++i) {
            final IdeDependency dep = deps[i];
            final File file = dep.getFile();
            if (file == null) {
                log.warn("No file given for " + dep);
                continue;
            }
            cp.add(file);
        }
        return cp.toArray(new File[cp.size()]);
    }

    protected void osgify(EclipseWriterConfig config) throws Exception {

        NuxeoAnalyzer analyzer = new NuxeoAnalyzer();
        final MavenProject project = config.getProject();
        analyzer.setJar(new File(project.getBuild().getOutputDirectory()));
        analyzer.setClasspath(getClasspath(config));
        addDirectives(analyzer, project.getArtifact(), false);
        if (config.isPde()) {
            File template = new File(config.getEclipseProjectDirectory().getPath() + "/src/main/resources/META-INF/MANIFEST.MF");
            if (template.exists()) {
                InputStream is = new FileInputStream(template);
                analyzer.mergeManifest(new Manifest(is));
            }
        }

        // version
        String version = analyzer.getProperty(Analyzer.BUNDLE_VERSION);
        if (version != null) {
            version = Builder.cleanupVersion(version);
            analyzer.setProperty(Analyzer.BUNDLE_VERSION, version);
        }

        // analyze
        try {
            Manifest manifest = analyzer.calcManifest();
            analyzer.addExportsVersion();
            File file = config.getManifest();
            if (file.exists()) {
                file.delete();
            }
            file.createNewFile();
            manifest.write(new FileOutputStream(config.getManifest()));
        } finally {
            analyzer.close();
        }

    }

    public boolean accept(Artifact artifact, Attributes attributes) {
        if (getDirectives(artifact) != null) {
            return true;
        }
        String bundleName = attributes.getValue("Bundle-Name");
        String symbolicName = attributes.getValue("Bundle-SymbolicName");
        if (symbolicName == null || bundleName != null) {
            return true;
        }
        return false;
    }

}
